/*
 * Copyright (C) 2005-2015 Alfresco Software Limited.
 *
 * This file is part of Alfresco
 *
 * Alfresco is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Alfresco is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with Alfresco. If not, see <http://www.gnu.org/licenses/>.
 */

package org.springframework.extensions.surf.mvc;

import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;

import org.springframework.extensions.surf.RequestContext;
import org.springframework.extensions.surf.UserFactory;
import org.springframework.extensions.surf.exception.WebScriptsPlatformException;
import org.springframework.extensions.surf.support.ThreadLocalRequestContext;
import org.springframework.extensions.surf.util.Base64;
import org.springframework.extensions.webscripts.connector.ConnectorService;
import org.springframework.extensions.webscripts.connector.CredentialVault;
import org.springframework.extensions.webscripts.connector.Credentials;
import org.springframework.extensions.webscripts.connector.CredentialsImpl;
import org.springframework.web.servlet.ModelAndView;

import java.io.IOException;

/**
 * Controller for feed requests. Feed requests are authenticated via Basic HTTP auth
 * to allow apps such as feed reader to access RSS feeds within a SURF app without
 * full authentication. It effectively authenticates the Guest users against the current
 * user session with the supplied Basic HTTP auth settings.
 * 
 * Any page requests made will still need to provide full authentication as required
 * by their respective endpoints. The feed controller is tied to the "alfresco-feed" endpoint.
 * 
 * @author Kevin Roast
 */
public class FeedController extends UrlViewController
{
    private static final String ENDPOINT_ALFRESCO_FEED = "alfresco-feed";
    
    protected ConnectorService connectorService;
    
    
    /**
     * Sets the connector service.
     * 
     * @param connectorService the new connector service
     */
    public void setConnectorService(ConnectorService connectorService)
    {
        this.connectorService = connectorService;
    }
    
    /* (non-Javadoc)
     * @see org.springframework.web.servlet.mvc.AbstractController#handleRequestInternal(javax.servlet.http.HttpServletRequest, javax.servlet.http.HttpServletResponse)
     */
    @Override
    protected ModelAndView handleRequestInternal(HttpServletRequest req, HttpServletResponse res)
    {
        // check for HTTP authorisation request (i.e. RSS feeds etc.)
        String authorization = req.getHeader("Authorization");
        if (authorization == null || authorization.length() == 0)
        {
            try {
                authorizedResponseStatus(res);
            } catch (IOException e) {
                throw new RuntimeException(e);
            }

            // no further processing as authentication is required but not provided
            // the browser will now prompt the user for appropriate credentials
            return null;
        }
        else
        {
            // user has provided authentication details with the request
            String[] authParts = authorization.split(" ");
            // test for a "negotiate" header - we will then suggest "basic" as the auth mechanism
            if (authParts[0].equalsIgnoreCase("negotiate"))
            {
                try {
                    authorizedResponseStatus(res);
                } catch (IOException e) {
                    throw new RuntimeException(e);
                }

                // no further processing as authentication is required but not provided
               // the browser will now prompt the user for appropriate credentials
               return null;
            }
            if (!authParts[0].equalsIgnoreCase("basic"))
            {
                throw new WebScriptsPlatformException("Authorization '" + authParts[0] + "' not supported.");
            }
            
            String[] values = new String(Base64.decode(authParts[1])).split(":");
            if (values.length == 2)
            {
                // assume username and password passed as the parts and
                // build an unauthenticated authentication connector then
                // apply the supplied credentials to it
                try
                {
                    // generate the credentials based on the auth details provided
                    Credentials credentials = new CredentialsImpl(ENDPOINT_ALFRESCO_FEED);
                    String username = values[0];
                    credentials.setProperty(Credentials.CREDENTIAL_USERNAME, username);
                    credentials.setProperty(Credentials.CREDENTIAL_PASSWORD, values[1]);
                    CredentialVault vault = connectorService.getCredentialVault(req.getSession(), username);
                    vault.store(credentials);
                    
                    // set USER_ID into the session
                    req.getSession().setAttribute(UserFactory.SESSION_ATTRIBUTE_KEY_USER_ID, username);
                    
                    // override endpoint that will be used to retrieve user details
                    RequestContext context = ThreadLocalRequestContext.getRequestContext();
                    context.getAttributes().put(RequestContext.USER_ENDPOINT, ENDPOINT_ALFRESCO_FEED);
                    
                    // set cache control for rss feeds - some feed readers *must* have a value set and it
                    // must not be no-cache!
                    res.setHeader("Cache-Control", "max-age=600, must-revalidate");
                }
                catch (Exception err)
                {
                    throw new WebScriptsPlatformException("Failed to provision connector.", err);
                }
            }
            else
            {
                try {
                    authorizedResponseStatus(res);
                } catch (IOException e) {
                    throw new RuntimeException(e);
                }

                return null;
            }
        }
        
        return super.handleRequestInternal(req, res);
    }
    
    private void authorizedResponseStatus(HttpServletResponse res) throws IOException {
        res.setHeader("WWW-Authenticate", "Basic realm=\"Alfresco\"");
        res.sendError(HttpServletResponse.SC_UNAUTHORIZED,
                "Requested endpoint requires authentication.");
    }
}
