/*
 * Decompiled with CFR 0.152.
 */
package org.alfresco.event.outbox;

import java.time.Duration;
import java.util.Optional;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Supplier;
import org.alfresco.event.outbox.EventProcessor;
import org.alfresco.event.outbox.EventTableStats;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class OutboxWorker {
    private final EventProcessor eventProcessor;
    private volatile boolean shouldBeRunning = false;
    private final AtomicReference<WorkerThread> workerThread = new AtomicReference();

    public OutboxWorker(EventProcessor eventProcessor) {
        this.eventProcessor = eventProcessor;
    }

    public void start() {
        this.shouldBeRunning = true;
        this.ensureWorkerIsInDesiredState();
    }

    public void stop() {
        this.shouldBeRunning = false;
        this.ensureWorkerIsInDesiredState();
    }

    public boolean isRunning() {
        WorkerThread currentWorkerThread = this.workerThread.get();
        return currentWorkerThread != null && currentWorkerThread.isRunning();
    }

    public void ensureWorkerIsInDesiredState() {
        WorkerThread currentThread = this.workerThread.get();
        if (currentThread == null) {
            if (this.shouldBeRunning) {
                this.startWorkerThread();
            }
        } else if (currentThread.isRunning()) {
            if (!this.shouldBeRunning) {
                this.stopWorkerThread(currentThread);
            }
        } else if (this.shouldBeRunning) {
            this.restartWorkerThread(currentThread);
        }
    }

    private void startWorkerThread() {
        WorkerThread newWorker = new WorkerThread(this.eventProcessor::processEvents);
        if (this.workerThread.compareAndSet(null, newWorker)) {
            newWorker.start();
        }
    }

    private void stopWorkerThread(WorkerThread thread) {
        thread.stop();
    }

    private void restartWorkerThread(WorkerThread thread) {
        thread.stop();
        WorkerThread newWorker = new WorkerThread(this.eventProcessor::processEvents);
        if (this.workerThread.compareAndSet(thread, newWorker)) {
            newWorker.start();
        }
    }

    EventTableStats getStats() {
        return this.eventProcessor.getStats();
    }

    static class WorkerThread {
        private static final Logger LOGGER = LoggerFactory.getLogger(WorkerThread.class);
        private static final long START_STOP_WAIT_MS = 100L;
        private static final AtomicInteger THREAD_ID_COUNTER = new AtomicInteger();
        private static final CountDownLatch startLatch = new CountDownLatch(1);
        private final Supplier<Duration> singleIteration;
        private final Thread thread;
        private volatile boolean stopRequested = false;
        private volatile Throwable failure;

        WorkerThread(Supplier<Duration> singleIteration) {
            this.singleIteration = singleIteration;
            this.thread = new Thread(this::workerLoop, "OutboxWorkerThread#" + THREAD_ID_COUNTER.incrementAndGet());
            this.thread.setDaemon(true);
        }

        public void start() {
            if (this.isRunning()) {
                return;
            }
            if (this.stopRequested || this.failure != null) {
                throw new IllegalStateException("Worker thread has been already stopped and cannot be restarted.");
            }
            this.thread.start();
            try {
                if (!startLatch.await(100L, TimeUnit.MILLISECONDS)) {
                    LOGGER.warn("Failed to wait for the worker thread ({}) to be started. Will continue anyway.", (Object)this.thread.getName());
                }
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        }

        public void stop() {
            this.stopRequested = true;
            this.thread.interrupt();
            try {
                this.thread.join(100L);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        }

        public boolean isRunning() {
            return this.thread.isAlive();
        }

        private void workerLoop() {
            startLatch.countDown();
            LOGGER.info("Worker has been started.");
            while (!this.stopRequested) {
                Duration delay;
                try {
                    LOGGER.debug("Executing single iteration.");
                    delay = this.singleIteration.get();
                }
                catch (RuntimeException e) {
                    LOGGER.debug("Recoverable exception occurred. Worker won't be stopped.", (Throwable)e);
                    continue;
                }
                catch (Error e) {
                    this.failure = e;
                    LOGGER.error("Unrecoverable error occurred. Worker will be stopped.", (Throwable)e);
                    break;
                }
                if (delay.isZero()) continue;
                try {
                    TimeUnit.NANOSECONDS.sleep(delay.toNanos());
                }
                catch (InterruptedException e) {
                    LOGGER.info("Worker thread has been interrupted. Worker will be stopped.");
                    Thread.currentThread().interrupt();
                    break;
                }
            }
            LOGGER.info("Worker has been stopped.");
        }

        public Optional<Throwable> getFailure() {
            return Optional.ofNullable(this.failure);
        }
    }
}

