/*
 * Copyright 2005-2015 Alfresco Software, Ltd.  All rights reserved.
 *
 * This file is part of a proprietary Alfresco module.
 *
 * License rights for this program are granted under the terms of the "Alfresco
 * Component License", which defines the permitted uses of the module.
 * License terms can be found in the file license.txt distributed with this module.
 */
package org.alfresco.officeservices.testclient.util;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.util.BitSet;

/**
 * This class is similar to the well known <code>java.net.URLEncoder</code>, but it does provide
 * a way to define the set of characters that should NOT be encoded.<br/>
 * This class has been adopted from org.apache.catalina.util.URLEncoder from Tomcat 6.0.35 under the
 * Apache license.
 * 
 * @author Stefan Kopf, xaldon Technologies GmbH.
 * 
 * @since 2.8
 */
public class URLEncoder
{

    /** the set of hex characters to be used  for encoding */
    static final char[] hexadecimal = { '0', '1', '2', '3', '4', '5', '6', '7', '8', '9', 'A', 'B', 'C', 'D', 'E', 'F' };

    /** set of safe  characters that are not encoded */
    protected BitSet safeCharacters = new BitSet(256);

    /**
     * Create a new URLEncoder and add the characters a-z, A-Z and 0-9 to the set of safe characters
     */
    public URLEncoder()
    {
        for (char i = 'a'; i <= 'z'; i++)
        {
            addSafeCharacter(i);
        }
        for (char i = 'A'; i <= 'Z'; i++)
        {
            addSafeCharacter(i);
        }
        for (char i = '0'; i <= '9'; i++)
        {
            addSafeCharacter(i);
        }
    }

    /**
     * Add the given character to the set of  safe characters
     * 
     * @param c the character to be added to the set of safe characters
     */
    public void addSafeCharacter(char c)
    {
        safeCharacters.set(c);
    }

    /**
     * Encode the given String by replacing all non safe characters by the sequence
     * starting with the percent character followed by the two digit hex representation
     * of that character.<br/>
     * The given path is represented as UTF-8.
     * 
     * @param path the path to be encoded
     * 
     * @return the encoded path
     */
    public String encode(String path)
    {
        int maxBytesPerChar = 10;
        //int caseDiff = ('a' - 'A');
        StringBuffer rewrittenPath = new StringBuffer(path.length());
        ByteArrayOutputStream buf = new ByteArrayOutputStream(maxBytesPerChar);
        OutputStreamWriter writer = null;
        try
        {
            writer = new OutputStreamWriter(buf, "UTF8");
        }
        catch (Exception e)
        {
            writer = new OutputStreamWriter(buf);
        }

        for (int i = 0; i < path.length(); i++)
        {
            int c = (int) path.charAt(i);
            if (safeCharacters.get(c))
            {
                rewrittenPath.append((char) c);
            }
            else
            {
                // convert to external encoding before hex conversion
                try
                {
                    writer.write((char) c);
                    writer.flush();
                }
                catch (IOException e)
                {
                    buf.reset();
                    continue;
                }
                byte[] ba = buf.toByteArray();
                for (int j = 0; j < ba.length; j++)
                {
                    // Converting each byte in the buffer
                    byte toEncode = ba[j];
                    rewrittenPath.append('%');
                    int low = (int) (toEncode & 0x0f);
                    int high = (int) ((toEncode & 0xf0) >> 4);
                    rewrittenPath.append(hexadecimal[high]);
                    rewrittenPath.append(hexadecimal[low]);
                }
                buf.reset();
            }
        }
        return rewrittenPath.toString();
    }

}
