/*
 * Copyright 2005-2015 Alfresco Software, Ltd.  All rights reserved.
 *
 * This file is part of a proprietary Alfresco module.
 *
 * License rights for this program are granted under the terms of the "Alfresco
 * Component License", which defines the permitted uses of the module.
 * License terms can be found in the file license.txt distributed with this module.
 */
package org.alfresco.officeservices.testclient.fpse;

import java.io.UnsupportedEncodingException;
import java.net.URI;
import java.net.URLEncoder;
import java.util.ArrayList;

public class FPSERequest
{

    public static final String FPSE_CHARSET = "UTF-8";

    public static final String PARAMETER_SEPARATOR = "&";

    public static final String NAME_VALUE_SEPARATOR = "=";
    
    protected URI serviceEndpoint;
    
    protected String methodName;
    
    protected String methodVersion;
    
    protected ArrayList<FPSERequestParameter> methodParameters = new ArrayList<FPSERequestParameter>();
    
    public FPSERequest(URI endpoint, String name, String version)
    {
        serviceEndpoint = endpoint;
        methodName = name;
        methodVersion = version;
    }
    
    public void addParameter(String name, String value)
    {
        methodParameters.add(new FPSERequestParameter(name, value));
    }
    
    public URI getServiceEndpoint()
    {
        return serviceEndpoint;
    }
    
    public FPSERequestEntity getRequestEntity()
    {
        try
        {
            StringBuilder parameterData = new StringBuilder();
            parameterData.append(URLEncoder.encode("method", FPSE_CHARSET));
            parameterData.append(NAME_VALUE_SEPARATOR);
            parameterData.append(URLEncoder.encode(methodName+":"+methodVersion, FPSE_CHARSET));
            for(final FPSERequestParameter param : methodParameters)
            {
                final String encodedName = URLEncoder.encode(param.getName(), FPSE_CHARSET);
                final String value = param.getValue();
                final String encodedValue = value != null ? URLEncoder.encode(value, FPSE_CHARSET) : "";
                parameterData.append(PARAMETER_SEPARATOR);
                parameterData.append(encodedName);
                parameterData.append(NAME_VALUE_SEPARATOR);
                parameterData.append(encodedValue);
            }
            byte[] parameterBytes = parameterData.toString().getBytes(FPSE_CHARSET);
            return new FPSERequestEntity(parameterBytes);
        }
        catch(UnsupportedEncodingException uee)
        {
            throw new RuntimeException("Platform does not support UTF-8 encoding");
        }
    }
    
}
