/*
 * Decompiled with CFR 0.152.
 */
package org.alfresco.transform.misc.transformers;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.StringWriter;
import java.io.Writer;
import java.util.HashMap;
import java.util.Map;
import org.alfresco.transform.misc.transformers.TextToPdfContentTransformer;
import org.apache.pdfbox.Loader;
import org.apache.pdfbox.pdmodel.PDDocument;
import org.apache.pdfbox.pdmodel.font.Standard14Fonts;
import org.apache.pdfbox.text.PDFTextStripper;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

public class TextToPdfContentTransformerTest {
    TextToPdfContentTransformer transformer = new TextToPdfContentTransformer();
    private static final String TEXT_WITH_A_BREVE = "G\u0103mbardella, Matthew, Corets, Ev\u0103";
    private static final String TEXT_WITHOUT_A_BREVE = "Gambardella, Matthew, Corets, Eva";

    @BeforeEach
    public void setUp() {
        this.transformer.setStandardFont("Times-Roman");
        this.transformer.setFontSize(20);
    }

    @Test
    public void testUnlimitedPages() throws Exception {
        this.transformTextAndCheckPageLength(-1);
    }

    @Test
    public void testLimitedTo1Page() throws Exception {
        this.transformTextAndCheckPageLength(1);
    }

    @Test
    public void testLimitedTo2Pages() throws Exception {
        this.transformTextAndCheckPageLength(2);
    }

    @Test
    public void testLimitedTo50Pages() throws Exception {
        this.transformTextAndCheckPageLength(50);
    }

    @Test
    public void test1UTF16BigEndianBomBigEndianChars() throws Exception {
        String expectedByteOrder = "fe ff 00 31 00 20 00 49";
        this.transformTextAndCheck("UTF-16", true, true, expectedByteOrder);
        this.transformTextAndCheck("UTF-16", true, true, expectedByteOrder);
        this.transformTextAndCheck("UTF-16BE", true, true, expectedByteOrder);
        this.transformTextAndCheck("UTF-16LE", true, true, expectedByteOrder);
    }

    @Test
    public void test2UTF16LittleEndianBomLittleEndianChars() throws Exception {
        this.transformTextAndCheck("UTF-16", false, true, "ff fe 31 00 20 00 49 00");
    }

    @Test
    public void test3UTF16NoBomBigEndianChars() throws Exception {
        this.transformTextAndCheck("UTF-16", true, null, "00 31 00 20 00 49");
    }

    @Test
    public void test4UTF16NoBomLittleEndianChars() throws Exception {
        this.transformTextAndCheck("UTF-16", false, null, "31 00 20 00 49 00");
    }

    @Test
    public void test5UTF16BigEndianBomLittleEndianChars() throws Exception {
        this.transformTextAndCheck("UTF-16", false, false, "fe ff 31 00 20 00 49 00");
    }

    @Test
    public void test6UTF16LittleEndianBomBigEndianChars() throws Exception {
        this.transformTextAndCheck("UTF-16", true, false, "ff fe 00 31 00 20 00 49");
    }

    @Test
    public void testUTF8WithBOM() throws Exception {
        TransformCheckResult result = this.transformTextAndCheck("UTF-8", null, true, "ef bb bf 31 20 49 20 6d");
        Assertions.assertEquals((Object)result.getUsedFont(), (Object)"Times-Roman");
    }

    @Test
    public void testUTF8WithoutBOM() throws Exception {
        this.transformTextAndCheck("UTF-8", null, false, "31 20 49 20 6d 75 73 74");
    }

    @Test
    public void testMNT23960_TimesBold_WithoutBreve() throws Exception {
        File sourceFile = File.createTempFile("TMP_Times-Bold", ".txt");
        String encoding = "UTF-8";
        this.writeToFile(sourceFile, TEXT_WITHOUT_A_BREVE, encoding, null, null);
        HashMap<String, String> parameters = new HashMap<String, String>();
        parameters.put("pdfFont", Standard14Fonts.FontName.TIMES_BOLD.getName());
        parameters.put("pdfFontSize", "30");
        TransformCheckResult result = this.transformTextAndCheck(sourceFile, encoding, TEXT_WITHOUT_A_BREVE, String.valueOf(-1), true, parameters, false);
        Assertions.assertEquals((Object)result.getUsedFont(), (Object)Standard14Fonts.FontName.TIMES_BOLD.getName());
        Assertions.assertNull((Object)result.getErrorMessage());
    }

    @Test
    public void testMNT23960_InexistentFont_WithoutBreve() throws Exception {
        File sourceFile = File.createTempFile("TMP_MyDummyFont", ".txt");
        String encoding = "UTF-8";
        this.writeToFile(sourceFile, TEXT_WITHOUT_A_BREVE, encoding, null, null);
        HashMap<String, String> parameters = new HashMap<String, String>();
        parameters.put("pdfFont", "MyDummyFont");
        TransformCheckResult result = this.transformTextAndCheck(sourceFile, encoding, TEXT_WITHOUT_A_BREVE, String.valueOf(-1), true, parameters, false);
        Assertions.assertEquals((Object)result.getUsedFont(), (Object)Standard14Fonts.FontName.TIMES_ROMAN.getName());
        Assertions.assertNull((Object)result.getErrorMessage());
    }

    @Test
    public void testMNT23960_TimesBold_WithBreve() throws Exception {
        File sourceFile = File.createTempFile("TMP_Times-Bold", ".txt");
        String encoding = "UTF-8";
        this.writeToFile(sourceFile, TEXT_WITH_A_BREVE, encoding, null, null);
        HashMap<String, String> parameters = new HashMap<String, String>();
        parameters.put("pdfFont", Standard14Fonts.FontName.TIMES_BOLD.getName());
        TransformCheckResult result = this.transformTextAndCheck(sourceFile, encoding, TEXT_WITH_A_BREVE, String.valueOf(-1), true, parameters, true);
        Assertions.assertEquals((Object)result.getUsedFont(), (Object)Standard14Fonts.FontName.TIMES_BOLD.getName());
        Assertions.assertNotNull((Object)result.getErrorMessage());
        Assertions.assertTrue((boolean)result.getErrorMessage().contains(Standard14Fonts.FontName.TIMES_BOLD.getName()));
    }

    protected TransformCheckResult transformTextAndCheck(String encoding, Boolean bigEndian, Boolean validBom, String expectedByteOrder) throws Exception {
        return this.transformTextAndCheckImpl(-1, encoding, bigEndian, validBom, expectedByteOrder);
    }

    protected TransformCheckResult transformTextAndCheckPageLength(int pageLimit) throws Exception {
        return this.transformTextAndCheckImpl(pageLimit, "UTF-8", null, null, null);
    }

    private TransformCheckResult transformTextAndCheckImpl(int pageLimit, String encoding, Boolean bigEndian, Boolean validBom, String expectedByteOrder) throws Exception {
        StringBuilder sb = new StringBuilder();
        String checkText = this.createTestText(pageLimit, sb);
        String text = sb.toString();
        File sourceFile = File.createTempFile("AlfrescoTestSource_", ".txt");
        this.writeToFile(sourceFile, text, encoding, bigEndian, validBom);
        this.checkFileBytes(sourceFile, expectedByteOrder);
        return this.transformTextAndCheck(sourceFile, encoding, checkText, String.valueOf(pageLimit));
    }

    private String createTestText(int pageLimit, StringBuilder sb) {
        int pageLength = 32;
        int lines = (pageLength + 10) * (pageLimit > 0 ? pageLimit : 1);
        String checkText = null;
        int cutoff = pageLimit * pageLength;
        for (int i = 1; i <= lines; ++i) {
            sb.append(Integer.toString(i));
            sb.append(" I must not talk in class or feed my homework to my cat.\n");
            if (i != cutoff) continue;
            checkText = sb.toString();
        }
        sb.append("\nBart\n");
        String text = sb.toString();
        checkText = checkText == null ? this.clean(text) : this.clean(checkText);
        return checkText;
    }

    private TransformCheckResult transformTextAndCheck(File sourceFile, String encoding, String checkText, String pageLimit) throws Exception {
        return this.transformTextAndCheck(sourceFile, encoding, checkText, pageLimit, true, null, false);
    }

    private TransformCheckResult transformTextAndCheck(File sourceFile, String encoding, String checkText, String pageLimit, boolean clean, Map<String, String> extraParameters, boolean shouldFail) throws Exception {
        TransformCheckResult result = new TransformCheckResult();
        File targetFile = File.createTempFile("AlfrescoTestTarget_", ".pdf");
        HashMap<String, String> parameters = new HashMap<String, String>();
        parameters.put("pageLimit", pageLimit);
        parameters.put("sourceEncoding", encoding);
        if (extraParameters != null) {
            parameters.putAll(extraParameters);
        }
        boolean failed = false;
        try {
            this.transformer.transform("text/plain", "application/pdf", parameters, sourceFile, targetFile, null);
        }
        catch (Exception e) {
            failed = true;
            result.setErrorMessage(e.getMessage());
        }
        result.setUsedFont(this.transformer.getUsedFont());
        if (!failed) {
            PDDocument doc = Loader.loadPDF((File)targetFile);
            PDFTextStripper textStripper = new PDFTextStripper();
            StringWriter textWriter = new StringWriter();
            textStripper.writeText(doc, (Writer)textWriter);
            doc.close();
            String roundTrip = this.clean(textWriter.toString());
            Assertions.assertEquals((Object)checkText, (Object)roundTrip, (String)("Incorrect text in PDF when starting from text in " + encoding));
        } else {
            Assertions.assertTrue((shouldFail && failed ? 1 : 0) != 0);
        }
        sourceFile.delete();
        targetFile.delete();
        return result;
    }

    private String clean(String text) {
        text = text.replaceAll("\\s+\\r", "");
        text = text.replaceAll("\\s+\\n", "");
        text = text.replaceAll("\\r", "");
        text = text.replaceAll("\\n", "");
        return text;
    }

    private void writeToFile(File file, String content, String encoding, Boolean bigEndian, Boolean validBom) throws Exception {
        File originalFile = file;
        if (bigEndian != null) {
            file = File.createTempFile("AlfrescoTestTmpSrc_", ".txt");
            encoding = "UTF-16";
        }
        try (OutputStreamWriter ow = new OutputStreamWriter((OutputStream)new FileOutputStream(file), encoding);){
            if (bigEndian == null && encoding != null && "UTF-8".equals(encoding.toUpperCase()) && validBom != null && validBom.booleanValue()) {
                ow.append("\ufeff");
            }
            ow.append(content);
        }
        if (bigEndian != null) {
            boolean firstRead = true;
            byte[] bytes = new byte[8192];
            try (BufferedInputStream is = new BufferedInputStream(new FileInputStream(file));
                 BufferedOutputStream os = new BufferedOutputStream(new FileOutputStream(originalFile));){
                int l;
                boolean switchBytes = false;
                do {
                    int len;
                    l = ((InputStream)is).read(bytes);
                    int off = 0;
                    if (firstRead) {
                        firstRead = false;
                        boolean actualEndianBytes = bytes[0] == -2;
                        boolean bl = switchBytes = actualEndianBytes != bigEndian;
                        if (validBom == null) {
                            off = 2;
                        } else if (!validBom.booleanValue()) {
                            byte aByte = bytes[0];
                            bytes[0] = bytes[1];
                            bytes[1] = aByte;
                        }
                    }
                    if ((len = l - off) <= 0) continue;
                    if (switchBytes) {
                        for (int i = 0; i < l; i += 2) {
                            byte aByte = bytes[i];
                            bytes[i] = bytes[i + 1];
                            bytes[i + 1] = aByte;
                        }
                    }
                    ((OutputStream)os).write(bytes, off, len - off);
                } while (l != -1);
            }
        }
    }

    private void checkFileBytes(File sourceFile, String expectedByteOrder) throws Exception {
        if (expectedByteOrder != null) {
            byte[] expectedBytes = this.hexToBytes(expectedByteOrder);
            int l = expectedBytes.length;
            byte[] actualBytes = new byte[l];
            FileInputStream is = new FileInputStream(sourceFile);
            is.read(actualBytes, 0, l);
            String actualByteOrder = this.bytesToHex(actualBytes);
            Assertions.assertEquals((Object)expectedByteOrder, (Object)actualByteOrder, (String)"The sourceFile does not contain the expected bytes");
        }
    }

    private byte[] hexToBytes(String hexString) {
        hexString = hexString.replaceAll(" *", "");
        int len = hexString.length() / 2;
        byte[] bytes = new byte[len];
        int j = 0;
        for (int i = 0; i < len; ++i) {
            int firstDigit = Character.digit(hexString.charAt(j++), 16);
            int secondDigit = Character.digit(hexString.charAt(j++), 16);
            bytes[i] = (byte)((firstDigit << 4) + secondDigit);
        }
        return bytes;
    }

    private String bytesToHex(byte[] bytes) {
        StringBuffer sb = new StringBuffer();
        int len = bytes.length;
        for (int i = 0; i < len; ++i) {
            if (sb.length() > 0) {
                sb.append(' ');
            }
            sb.append(Character.forDigit(bytes[i] >> 4 & 0xF, 16));
            sb.append(Character.forDigit(bytes[i] & 0xF, 16));
        }
        return sb.toString();
    }

    private static class TransformCheckResult {
        private String usedFont;
        private String errorMessage;

        private TransformCheckResult() {
        }

        public String getUsedFont() {
            return this.usedFont;
        }

        public void setUsedFont(String usedFont) {
            this.usedFont = usedFont;
        }

        public String getErrorMessage() {
            return this.errorMessage;
        }

        public void setErrorMessage(String errorMessage) {
            this.errorMessage = errorMessage;
        }
    }
}

