/*
 * Decompiled with CFR 0.152.
 */
package org.alfresco.util;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.time.Duration;
import java.util.concurrent.atomic.AtomicLong;
import org.alfresco.api.AlfrescoPublicApi;
import org.alfresco.error.AlfrescoRuntimeException;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.quartz.Job;
import org.quartz.JobExecutionContext;
import org.quartz.JobExecutionException;

@AlfrescoPublicApi
public class TempFileProvider {
    private static final int BUFFER_SIZE = 40960;
    public static final String ALFRESCO_TEMP_FILE_DIR = "Alfresco";
    public static final String ALFRESCO_LONG_LIFE_FILE_DIR = "longLife";
    public static final String SYSTEM_KEY_TEMP_DIR = "java.io.tmpdir";
    private static final Log logger = LogFactory.getLog(TempFileProvider.class);
    private static int MAX_RETRIES = 3;

    private TempFileProvider() {
    }

    public static File getSystemTempDir() {
        String systemTempDirPath = System.getProperty(SYSTEM_KEY_TEMP_DIR);
        if (systemTempDirPath == null) {
            throw new AlfrescoRuntimeException("System property not available: java.io.tmpdir");
        }
        File systemTempDir = new File(systemTempDirPath);
        if (logger.isDebugEnabled()) {
            logger.debug((Object)("Created system temporary directory: " + systemTempDir));
        }
        return systemTempDir;
    }

    public static File getTempDir() {
        return TempFileProvider.getTempDir(ALFRESCO_TEMP_FILE_DIR);
    }

    public static File getTempDir(String dirName) {
        File systemTempDir = TempFileProvider.getSystemTempDir();
        File tempDir = new File(systemTempDir, dirName);
        if (!tempDir.exists()) {
            if (!tempDir.mkdirs()) {
                if (!tempDir.exists()) {
                    throw new AlfrescoRuntimeException("Failed to create temp directory: " + tempDir);
                }
            } else if (logger.isDebugEnabled()) {
                logger.debug((Object)("Created temp directory: " + tempDir));
            }
        }
        return tempDir;
    }

    public static File getLongLifeTempDir(String key) {
        String folderName = "longLife_" + key;
        File tempDir = TempFileProvider.getTempDir();
        File longLifeDir = new File(tempDir, folderName);
        if (longLifeDir.exists()) {
            if (logger.isDebugEnabled()) {
                logger.debug((Object)("Already exists: " + longLifeDir));
            }
            return longLifeDir;
        }
        for (int retry = 0; retry < MAX_RETRIES; ++retry) {
            boolean created = longLifeDir.mkdirs();
            if (created) {
                if (logger.isDebugEnabled()) {
                    logger.debug((Object)("Created long life temp directory: " + longLifeDir));
                }
                return longLifeDir;
            }
            if (!longLifeDir.exists()) continue;
            if (logger.isDebugEnabled()) {
                logger.debug((Object)("Another thread created long life temp directory: " + longLifeDir));
            }
            return longLifeDir;
        }
        throw new AlfrescoRuntimeException("Failed to create temp directory: " + longLifeDir);
    }

    public static File createTempFile(InputStream in, String namePrefix, String nameSufix) throws Exception {
        if (null == in) {
            return null;
        }
        File file = TempFileProvider.createTempFile(namePrefix, nameSufix);
        BufferedOutputStream out = new BufferedOutputStream(new FileOutputStream(file), 40960);
        try {
            int i;
            byte[] buffer = new byte[40960];
            while ((i = in.read(buffer)) > -1) {
                ((OutputStream)out).write(buffer, 0, i);
            }
        }
        catch (Exception e) {
            file.delete();
            throw e;
        }
        finally {
            in.close();
            ((OutputStream)out).flush();
            ((OutputStream)out).close();
        }
        return file;
    }

    private static boolean isLongLifeTempDir(File file) {
        if (file.isDirectory()) {
            return file.getName().startsWith(ALFRESCO_LONG_LIFE_FILE_DIR);
        }
        return false;
    }

    public static File createTempFile(String prefix, String suffix) {
        File tempDir = TempFileProvider.getTempDir();
        return TempFileProvider.createTempFile(prefix, suffix, tempDir);
    }

    public static File createTempFile(String prefix, String suffix, File directory) {
        try {
            File tempFile = File.createTempFile(prefix, suffix, directory);
            if (logger.isDebugEnabled()) {
                logger.debug((Object)("Creating tmp file: " + tempFile));
            }
            return tempFile;
        }
        catch (IOException e) {
            throw new AlfrescoRuntimeException("Failed to created temp file: \n   prefix: " + prefix + "\n   suffix: " + suffix + "\n   directory: " + directory, e);
        }
    }

    @AlfrescoPublicApi
    public static class TempFileCleanerJob
    implements Job {
        public static final String KEY_PROTECT_HOURS = "protectHours";
        public static final String KEY_DIRECTORY_NAME = "directoryName";
        public static final String KEY_MAX_FILES_TO_DELETE = "maxFilesToDelete";
        public static final String KEY_MAX_TIME_TO_RUN = "maxTimeToRun";
        private static long jobStartTime;
        private static AtomicLong maxFilesToDelete;
        private static Duration maxTimeToRun;

        public void execute(JobExecutionContext context) throws JobExecutionException {
            String strProtectHours = (String)context.getJobDetail().getJobDataMap().get((Object)KEY_PROTECT_HOURS);
            if (strProtectHours == null) {
                throw new JobExecutionException("Missing job data: protectHours");
            }
            int protectHours = -1;
            try {
                protectHours = Integer.parseInt(strProtectHours);
            }
            catch (NumberFormatException e) {
                throw new JobExecutionException("Invalid job data protectHours: " + strProtectHours);
            }
            if (protectHours < 0 || protectHours > 8760) {
                throw new JobExecutionException("Hours to protect temp files must be 0 <= x <= 8760");
            }
            String directoryName = (String)context.getJobDetail().getJobDataMap().get((Object)KEY_DIRECTORY_NAME);
            try {
                Object oMaxFilesToDelete = context.getJobDetail().getJobDataMap().get((Object)KEY_MAX_FILES_TO_DELETE);
                if (oMaxFilesToDelete != null) {
                    String strMaxFilesToDelete = (String)oMaxFilesToDelete;
                    maxFilesToDelete = new AtomicLong(Long.parseLong(strMaxFilesToDelete));
                    logger.debug((Object)("Set the maximum number of temp files to be deleted to: " + maxFilesToDelete.get()));
                } else {
                    logger.debug((Object)"No maximum number of files was configured for the temp file clean job.");
                }
            }
            catch (Exception e) {
                logger.warn((Object)e);
                throw new JobExecutionException("Invalid job data, maxFilesToDelete");
            }
            try {
                Object oMaxTimeToRun = context.getJobDetail().getJobDataMap().get((Object)KEY_MAX_TIME_TO_RUN);
                if (oMaxTimeToRun != null) {
                    String strMaxTimeToRun = (String)oMaxTimeToRun;
                    maxTimeToRun = Duration.parse(strMaxTimeToRun);
                    logger.debug((Object)("Set the maximum duration time of the temp file clean job to: " + maxTimeToRun));
                } else {
                    logger.debug((Object)"No maximum duration was configured for the temp file clean job.");
                }
            }
            catch (Exception e) {
                logger.warn((Object)e);
                throw new JobExecutionException("Invalid job data, maxTimeToRun");
            }
            if (directoryName == null) {
                directoryName = TempFileProvider.ALFRESCO_TEMP_FILE_DIR;
            }
            jobStartTime = System.currentTimeMillis();
            long aFewHoursBack = jobStartTime - 3600000L * (long)protectHours;
            long aLongTimeBack = jobStartTime - 86400000L;
            File tempDir = TempFileProvider.getTempDir(directoryName);
            int count = TempFileCleanerJob.removeFiles(tempDir, aFewHoursBack, aLongTimeBack, false);
            logger.debug((Object)("Removed " + count + " files from temp directory: " + tempDir));
        }

        public static int removeFiles(long removeBefore) {
            File tempDir = TempFileProvider.getTempDir();
            return TempFileCleanerJob.removeFiles(tempDir, removeBefore, removeBefore, false);
        }

        private static int removeFiles(File directory, long removeBefore, long longLifeBefore, boolean removeDir) {
            if (!directory.isDirectory()) {
                throw new IllegalArgumentException("Expected a directory to clear: " + directory);
            }
            if (!directory.exists()) {
                return 0;
            }
            File[] files = directory.listFiles();
            File[] filesToIterate = files != null ? files : new File[]{};
            int count = 0;
            for (File file : filesToIterate) {
                if (TempFileCleanerJob.shouldTheDeletionStop()) break;
                if (file.isDirectory()) {
                    int countRemoved = TempFileCleanerJob.removeFiles(file, TempFileProvider.isLongLifeTempDir(file) ? longLifeBefore : removeBefore, longLifeBefore, true);
                    logger.debug((Object)("Removed " + countRemoved + " files from " + (TempFileProvider.isLongLifeTempDir(file) ? "temp " : " ") + "directory: " + file));
                    continue;
                }
                if (file.lastModified() > removeBefore) continue;
                try {
                    logger.debug((Object)("Deleting temp file: " + file));
                    file.delete();
                    if (maxFilesToDelete != null) {
                        maxFilesToDelete.decrementAndGet();
                        logger.debug((Object)(maxFilesToDelete.get() + " files left to delete."));
                    }
                    if (maxTimeToRun != null) {
                        logger.debug((Object)(jobStartTime + maxTimeToRun.toMillis() - System.currentTimeMillis() + " millis left to delete."));
                    }
                    ++count;
                }
                catch (Throwable e) {
                    logger.info((Object)("Failed to remove temp file: " + file));
                }
            }
            if (removeDir) {
                try {
                    File[] listing = directory.listFiles();
                    if (listing != null && listing.length == 0) {
                        logger.debug((Object)("Deleting empty directory: " + directory));
                        directory.delete();
                    }
                }
                catch (Throwable e) {
                    logger.info((Object)("Failed to remove temp directory: " + directory), e);
                }
            }
            return count;
        }

        private static boolean shouldTheDeletionStop() {
            return maxFilesToDelete != null && maxFilesToDelete.get() <= 0L || maxTimeToRun != null && jobStartTime + maxTimeToRun.toMillis() < System.currentTimeMillis();
        }
    }
}

