/*
 * Copyright (C) 2005-2012 Alfresco Software Limited.
 *
 * This file is part of Alfresco
 *
 * Alfresco is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Alfresco is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with Alfresco. If not, see <http://www.gnu.org/licenses/>.
 */
package org.alfresco.bm.user;

import java.util.Iterator;
import java.util.List;

import org.springframework.beans.factory.InitializingBean;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Sort;
import org.springframework.data.domain.Sort.Direction;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;

import com.mongodb.BasicDBObjectBuilder;
import com.mongodb.DBObject;
import com.mongodb.WriteConcern;

/**
 * Service providing access to {@link UserData} storage. All {@link UserData} returned from and persisted
 * with this service will be testrun-specific. The testrun-identifier is set in the constructor.
 *
 * @author Frederik Heremans
 * @author Derek Hulley
 * @author steveglover
 * @since 1.1
 */
public class UserDataServiceImpl extends AbstractUserDataService implements InitializingBean
{
    private MongoTemplate mongo;
    private String collectionName;
    
    public UserDataServiceImpl(MongoTemplate mongo, String collectionName)
    {
        this.mongo = mongo;
        this.collectionName = collectionName;
    }
    
    @Override
    public void afterPropertiesSet() throws Exception
    {
        checkIndexes();
    }

    /**
     * Ensure that the MongoDB collection has the required indexes associated with
     * this user bean.
     * 
     * @param mongo                 connection to MongoDB
     * @param collectionName        name of DB collection containing data
     */
    private void checkIndexes()
    {
        mongo.getDb().getCollection(collectionName).setWriteConcern(WriteConcern.SAFE);
        
        DBObject uidxUserName = BasicDBObjectBuilder
                .start(UserData.FIELD_USERNAME, 1)
                .get();
        mongo.getDb().getCollection(collectionName).ensureIndex(uidxUserName, "uidx_username", true);

        DBObject uidxEmail = BasicDBObjectBuilder
                .start(UserData.FIELD_EMAIL, 1)
                .get();
        mongo.getDb().getCollection(collectionName).ensureIndex(uidxEmail, "uidx_email", true);

        DBObject idxDomain = BasicDBObjectBuilder
                .start(UserData.FIELD_DOMAIN, 1)
                .get();
        mongo.getDb().getCollection(collectionName).ensureIndex(idxDomain, "idx_domain", false);
        
        DBObject idxCreated = BasicDBObjectBuilder
                .start(UserData.FIELD_CREATED, 1)
                .add(UserData.FIELD_RANDOMIZER, 2)
                .add(UserData.FIELD_DOMAIN, 3)
                .get();
        mongo.getDb().getCollection(collectionName).ensureIndex(idxCreated, "idx_created", false);
        
        DBObject idxCloudSignUp = BasicDBObjectBuilder
                .start(UserData.FIELD_CLOUD_SIGNUP, 1)
                .add(UserData.FIELD_CREATED, 2)
                .add(UserData.FIELD_RANDOMIZER, 3)
                .get();
        mongo.getDb().getCollection(collectionName).ensureIndex(idxCloudSignUp, "idx_cloudsignup", false);
        
        DBObject idxCloudSignUpId = BasicDBObjectBuilder
                .start(UserData.FIELD_CLOUD_SIGNUP + "." + CloudSignUpData.FIELD_ID, 1)
                .get();
        mongo.getDb().getCollection(collectionName).ensureIndex(idxCloudSignUpId, "idx_cloudsignup_id", false);
    }

    // TODO: Remove unused code
//    private Criteria getUserCriteria(boolean created, String networkId, Set<String> exclusions)
//    {
//        Criteria criteria = Criteria.where(UserData.FIELD_CREATED).is(created);
//        if(networkId != null)
//        {
//        	criteria = criteria.and(UserData.FIELD_DOMAIN).is(networkId);
//        }
//        if(exclusions != null && exclusions.size() > 0)
//        {
//            for(String excludeUserId : exclusions)
//            {
//            	criteria = criteria.and(UserData.FIELD_EMAIL).ne(excludeUserId);
//            }
//        }
//        return criteria;
//    }
//    
    /**
     * {@inheritDoc}
     */
    @Override
    public void createNewUser(UserData data)
    {
        mongo.insert(data, collectionName);
    }
    
    /**
     * {@inheritDoc}
     */
    @Override
    public void setUserTicket(String username, String ticket)
    {
        Criteria criteria = Criteria.where(UserData.FIELD_USERNAME).is(username);
        Query query = new Query(criteria);

        Update update = Update.update(UserData.FIELD_TICKET, ticket);
        mongo.updateFirst(query, update, collectionName);
    }
    
    /**
     * {@inheritDoc}
     */
    @Override
    public void setUserPassword(String username, String password)
    {
        Criteria criteria = Criteria.where(UserData.FIELD_USERNAME).is(username);
        Query query = new Query(criteria);

        Update update = Update.update(UserData.FIELD_PASSWORD, password);
        mongo.updateFirst(query, update, collectionName);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void setUserNodeId(String username, String nodeId)
    {
        Criteria criteria = Criteria.where(UserData.FIELD_USERNAME).is(username);
        Query query = new Query(criteria);

        Update update = Update.update(UserData.FIELD_NODE_ID, nodeId);
        mongo.updateFirst(query, update, collectionName);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void setUserCreated(String username, boolean created)
    {
        Criteria criteria = Criteria.where(UserData.FIELD_USERNAME).is(username);
        Query query = new Query(criteria);

        Update update = Update.update(UserData.FIELD_CREATED, created);
        mongo.updateFirst(query, update, collectionName);
    }
    
    /**
     * @param created               <tt>true</tt> to only count users present in Alfresco
     */
    @Override
    public long countUsers(boolean created)
    {
        long count = 0L;
        if (created)
        {
            Criteria usersCriteria = Criteria.where(UserData.FIELD_CREATED).is(Boolean.valueOf(created));
            Query usersQuery = new Query(usersCriteria);
            count = mongo.count(usersQuery, collectionName);
        }
        else
        {
            Query usersQuery = new Query();
            count = mongo.count(usersQuery, collectionName);
        }
        // Done
        return count;
    }

    /**
     * @return                      a count of all users in any state
     */
    @Override
    public long countUsers()
    {
        Query usersQuery = new Query();
        long count = mongo.count(usersQuery, collectionName);
        // Done
        return count;
    }
    
    /**
     * Find a user by username
     * 
     * @return                          the {@link UserData} found otherwise <tt>null</tt.
     */
    @Override
    public UserData findUserByUsername(String username)
    {
        Criteria criteria = Criteria.where(UserData.FIELD_USERNAME).is(username);
        Query query = new Query(criteria);
        return mongo.findOne(query, UserData.class, collectionName);
    }
    
    /**
     * Find a user by email address
     * 
     * @return                          the {@link UserData} found otherwise <tt>null</tt.
     */
    @Override
    public UserData findUserByEmail(String email)
    {
        Criteria criteria = Criteria.where(UserData.FIELD_EMAIL).is(email);
        Query query = new Query(criteria);
        return mongo.findOne(query, UserData.class, collectionName);
    }
    
    /**
     * @param created               <tt>true</tt> to only count users present in Alfresco
     */
    protected List<UserData> getUsers(boolean created, int startIndex, int count)
    {
        Criteria usersCriteria = Criteria.where(UserData.FIELD_CREATED).is(Boolean.valueOf(created));
        Query usersQuery = new Query(usersCriteria)
                .with(new Sort(UserData.FIELD_RANDOMIZER));
        usersQuery.skip(startIndex).limit(count);

        List<UserData> users = mongo.find(usersQuery, UserData.class, collectionName);
        
        // Done
        return users;
    }

    /*
     * CLOUD USER SERVICES
     */

    /**
     * Set the registration data for a user
     * 
     * @param username                  the username
     * @param cloudSignUp               the new registration data to set
     */
    @Override
    public void setUserCloudSignUp(String username, CloudSignUpData cloudSignUp)
    {
        Criteria criteria = Criteria.where(UserData.FIELD_USERNAME).is(username);
        Query query = new Query(criteria);

        Update update = Update.update(UserData.FIELD_CLOUD_SIGNUP, cloudSignUp);
        mongo.updateFirst(query, update, collectionName);
    }
    
    /**
     * Count the number of cloud-enabled users, regardless of signup state
     * 
     * @return              the number of users that have cloud registration details, regardless of state
     */
    @Override
    public long countCloudAwareUsers()
    {
        Criteria criteriaRegExists = Criteria.where(UserData.FIELD_CLOUD_SIGNUP).exists(true);
        Query queryRegExists = new Query(criteriaRegExists);
        return mongo.count(queryRegExists, collectionName);
    }
    
    /**
     * Retrieves a selection of users that have no cloud signup details.  Note they must also
     * not be created in any instance of Alfresco.
     */
    @Override
    public List<UserData> getUsersWithoutCloudSignUp(int startIndex, int count)
    {
        Criteria usersCriteria = Criteria
                .where(UserData.FIELD_CLOUD_SIGNUP).exists(Boolean.FALSE)
                .and(UserData.FIELD_CREATED).is(Boolean.FALSE);
        Query usersQuery = new Query(usersCriteria)
                .with(new Sort(UserData.FIELD_RANDOMIZER));
        usersQuery.skip(startIndex).limit(count);

        List<UserData> users = mongo.find(usersQuery, UserData.class, collectionName);
        
        // Done
        return users;
    }
    
    @Override
    public List<UserData> getUsersInDomain(String domain, int startIndex, int count)
    {
        Criteria criteriaDomain = Criteria.where(UserData.FIELD_DOMAIN).is(domain);
        Query queryUsersInDomain = new Query(criteriaDomain).skip(startIndex).limit(count);
        List<UserData> users = mongo.find(queryUsersInDomain, UserData.class, collectionName);
        return users;
    }
    
    @Override
    public Iterator<String> networksIterator()
    {
        @SuppressWarnings("unchecked")
        Iterator<String> networksIt = (Iterator<String>) mongo.getCollection(collectionName).distinct(UserData.FIELD_DOMAIN).iterator();
        return networksIt;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public UserData findUserByUserName(String userName)
    {
        Criteria criteria = Criteria.where(UserData.FIELD_USERNAME).is(userName);
        Query query = new Query(criteria);
        return mongo.findOne(query, UserData.class, collectionName);
    }

    /**
     * @return              the maximum value of the randomizer
     */
    private int getMaxRandomizer()
    {
        Criteria criteria = Criteria
                .where(UserData.FIELD_CREATED).is(Boolean.TRUE);
        Query query = new Query(criteria)
                .with(new Sort(Direction.DESC, UserData.FIELD_RANDOMIZER));
        query.limit(1);
        
        UserData firstResult = (UserData) mongo.findOne(query, UserData.class, collectionName);
        return firstResult == null ? 0 : firstResult.getRandomizer();
    }

    @Override
    public UserData randomUser()
    {
        int upper = getMaxRandomizer();             // The upper limit will be exclusive
        int random = (int) (Math.random() * (double) upper);
        
        Criteria criteria = Criteria
                .where(UserData.FIELD_CREATED).is(Boolean.TRUE)
                .and(UserData.FIELD_RANDOMIZER).gte(new Integer(random));
        Query query = new Query(criteria)
                .with(new Sort(UserData.FIELD_CREATED, UserData.FIELD_RANDOMIZER))
                .with(new PageRequest(0, 1));
        
        UserData firstResult = (UserData) mongo.findOne(query, UserData.class, collectionName);
        return firstResult;
    }
    // TODO: Remove unused code
//    
//    @Override
//    public UserData randomUser(String domain, Set<String> exclusions)
//    {
//        int upper = getMaxRandomizer();             // The upper limit will be exclusive
//        
//        Criteria criteria = Criteria
//                .where(UserData.FIELD_CREATED).is(Boolean.TRUE)
//                .and(UserData.FIELD_RANDOMIZER).gte(new Integer(upper))
//                .and(UserData.FIELD_DOMAIN).is(domain);
//        if (exclusions != null)
//        {
//            for (String excludeUserId : exclusions)
//            {
//                criteria = criteria.and(UserData.FIELD_EMAIL).ne(excludeUserId);
//            }
//        }
//        Query query = new Query(criteria);
//        query.sort().on(UserData.FIELD_CREATED, Order.ASCENDING).on(UserData.FIELD_RANDOMIZER, Order.ASCENDING);
//        query.limit(1);
//        
//        UserData firstResult = (UserData) mongo.findOne(query, UserData.class, collectionName);
//        return firstResult;
//    }
}
