package org.alfresco.bm.user;

import java.util.Iterator;
import java.util.List;

/**
 * Service providing access to {@link UserData} storage. All {@link UserData} returned from and persisted
 * with this service will be testrun-specific. The testrun-identifier is set in the constructor.
 *
 * @author Frederik Heremans
 * @author Derek Hulley
 * @author steveglover
 * @since 1.1
 */
public interface UserDataService
{
    public interface UserCallback
    {
        boolean callback(UserData user);
    };
    
    public void createNewUser(UserData data);

    /**
     * Store an authentication token (ticket) against a username
     */
    public void setUserTicket(String username, String ticket);
    
    /**
     * Update a user's password
     */
    public void setUserPassword(String username, String password);

    /**
     * Store a node ID associated with the username
     */
    public void setUserNodeId(String username, String nodeId);

    /**
     * Change the 'created' state of the user i.e. whether the user exists on the server or not
     */
    public void setUserCreated(String username, boolean created);
    
    /**
     * @param created               <tt>true</tt> to only count users present in Alfresco
     */
    public long countUsers(boolean created);

    /**
     * @return                      a count of all users in any state
     */
    public long countUsers();
    
    /**
     * Find a user by username
     * 
     * @return                          the {@link UserData} found otherwise <tt>null</tt>.
     */
    public UserData findUserByUsername(String username);
    
    /**
     * Find a user by email address
     * 
     * @return                          the {@link UserData} found otherwise <tt>null</tt>.
     */
    public UserData findUserByEmail(String email);
    
    /**
     * Get a list of usernames that are NOT created in alfresco with paging
     * 
     * @param startIndex index to start getting users from  
     * @param count number of users to fetch
     * @return      List of user data, which may be empty or less than the required count
     */
    public List<UserData> getUsersPendingCreation(int startIndex, int count);

    /**
     * Get a list of usernames that are created in alfresco with paging
     * 
     * @param startIndex    index to start getting users from  
     * @param count         number of users to fetch
     * @return              List of user data, which may be empty or less than the required count
     */
    public List<UserData> getCreatedUsers(int startIndex, int count);
    
    /*
     * CLOUD USER SERVICES
     */

    /**
     * Set the registration data for a user
     * 
     * @param username                  the username
     * @param cloudSignUp               the new registration data to set
     */
    public void setUserCloudSignUp(String username, CloudSignUpData cloudSignUp);
    
    /**
     * Count the number of cloud-enabled users, regardless of signup state
     * 
     * @return              the number of users that have cloud registration details, regardless of state
     */
    public long countCloudAwareUsers();
    
    /**
     * Retrieves a selection of users that have no cloud signup details.  Note they must also
     * not be created in any instance of Alfresco.
     */
    public List<UserData> getUsersWithoutCloudSignUp(int startIndex, int count);
    
    /**
     * Return a maximum of "max" users in the network with id "networkId".
     * 
     * @param domain                    the user domain
     * @param startIndex                the start index for paging
     * @param count                     the number of users to retrieve
     * 
     * @return a list of users in the domain
     */
    public List<UserData> getUsersInDomain(String domain, int startIndex, int count);

    /**
     * An iterator over networks in the users collection.
     * 
     * @return an iterator over networks
     */
    public Iterator<String> networksIterator();
    
    /**
     * Find a user by it's username.
     * @return the {@link UserData} found. Null, if user doesn't exist.
     */
    public UserData findUserByUserName(String userName);

    /**
     * Select a random, pre-created user.
     * <p/>
     * Note that this is useful only for large numbers of users.
     * 
     * @return      a random user or <tt>null</tt> if none are available
     */
    public UserData randomUser();
//
//    /**
//     * Select a random, pre-created user from a given tenant domain,
//     * excluding those users in the exclusion list.
//     * <p/>
//     * Note that this is useful only for large numbers of users.
//     * 
//     * @return      a random user from the domain or <tt>null</tt> if none are available
//     */
//    public UserData randomUser(String domain, Set<String> exclusions);
}
