/*
 * Copyright (C) 2005-2012 Alfresco Software Limited.
 *
 * This file is part of Alfresco
 *
 * Alfresco is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Alfresco is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with Alfresco. If not, see <http://www.gnu.org/licenses/>.
 */
package org.alfresco.bm.user;

import org.alfresco.bm.event.AbstractEventProcessor;
import org.alfresco.bm.event.Event;
import org.alfresco.bm.event.EventResult;

/**
 * Prepare a specific number of users for interaction with Alfresco.
 * This does not actually create users in Alfresco but merely creates
 * a population of users (email addresses, names, etc) that can be used
 * by subsequent operations.</br>
 * User data will be similar to this:<pre>
 *      {
 *        "_id" : ObjectId("504f19ce4ece382c484d3dc1"),
 *        "_class" : "org.alfresco.bm.user.UserData",
 *        "randomizer" : 774971,
 *        "username" : "0000001.test@00000.test",
 *        "password" : "0000001.test@00000.test",
 *        "created" : false,
 *        "firstName" : "0000001",
 *        "lastName" : "Test",
 *        "email" : "0000001.test@00000.test",
 *        "domain" : "00000.test",
 *        "cloudSignUp" :
 *        {
 *          "_id" : "activiti$10056",
 *          "key" : "2ef295d9-4c5e-45d2-b56a-a6693a8d36c9",
 *          "complete" : true
 *        }
 *      }
 * </pre> where the "randomizer" field is used to return users in a random order when
 * querying for lists of users.
 * 
 * <h1>Input</h1>
 * 
 * None.
 * 
 * <h1>Data</h1>
 * 
 * No data requirements.
 * 
 * <h1>Actions</h1>
 * 
 * Users data objects are created.  This data is local only.
 * 
 * <h1>Output</h1>
 * {@link #EVENT_NAME_USERS_PREPARED}: Marker indicating completion
 *
 * @author Derek Hulley
 * @since 1.1
 */
public class PrepareUsers extends AbstractEventProcessor
{
    public static final long DEFAULT_USERS_PER_DOMAIN = 100L;
    public static final String EVENT_NAME_USERS_PREPARED = "usersPrepared";
    
    private UserDataService userDataService;
    private long numberOfUsers;
    private long usersPerDomain;
    private String eventNameUsersPrepared;
    
    /**
     * @param userDataService       service for {@link UserData} operations
     * @param numberOfUsers         number of users to create in total
     */
    public PrepareUsers(UserDataService userDataService, long numberOfUsers)
    {
        this.userDataService = userDataService;
        this.numberOfUsers = numberOfUsers;
        this.usersPerDomain = DEFAULT_USERS_PER_DOMAIN;
        this.eventNameUsersPrepared = EVENT_NAME_USERS_PREPARED;
    }

    /**
     * Override the {@link #DEFAULT_USERS_PER_DOMAIN default} number of users per email domain
     */
    public void setUsersPerDomain(long usersPerDomain)
    {
        this.usersPerDomain = usersPerDomain;
    }

    /**
     * Override the {@link #EVENT_NAME_USERS_PREPARED default} event name when users have been prepared.
     */
    public void setEventNameUsersCreated(String eventNameUsersPrepared)
    {
        this.eventNameUsersPrepared = eventNameUsersPrepared;
    }

    public EventResult processEvent(Event event) throws Exception
    {
        // How many users must we create?
        long userCount = userDataService.countUsers(true);
        long usersToCreate = numberOfUsers - userCount;

        // How many domains must we create?
        long domainsToCreate = (long) (Math.ceil((double)numberOfUsers/(double)usersPerDomain));
        
        long count = 0L;
        // Loop over domains and users and make sure they exist
        for (long i = 0L; i < domainsToCreate; i++)
        {
            String domain = String.format("%05d", i) + ".example.com";
            for (int j = 0; count < usersToCreate && j < usersPerDomain; j++)
            {
                String firstName = String.format("%07d", count);
                String lastName = "Test";
                String email = (firstName + "." + lastName + "@" + domain).toLowerCase();
                String username = email;
                String password = username;
                // Does the user exist (by email)
                UserData user = userDataService.findUserByUsername(username);
                if (user != null)
                {
                    // User already exists
                    continue;
                }
                // Create data
                user = new UserData();
                user.setCreated(false);
                user.setCloudSignUp(null);
                user.setDomain(domain);
                user.setEmail(email);
                user.setFirstName(firstName);
                user.setLastName(lastName);
                user.setNodeId(null);
                user.setPassword(password);
                user.setTicket(null);
                user.setUsername(username);
                // Persist
                userDataService.createNewUser(user);
                count++;
            }
        }
        // Raise an event saying we're done
        String msg = "Created " + count + " users.";
        Event doneEvent = new Event(eventNameUsersPrepared, System.currentTimeMillis(), msg);
        // Done
        EventResult result = new EventResult("", doneEvent);
        return result;
    }
}
