/*
 * Copyright (C) 2005-2012 Alfresco Software Limited.
 *
 * This file is part of Alfresco
 *
 * Alfresco is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Alfresco is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with Alfresco. If not, see <http://www.gnu.org/licenses/>.
 */
package org.alfresco.bm.tools;

import java.util.Arrays;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.apache.commons.cli.PosixParser;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * Import classes off the classpath into a ZooKeeper location
 * 
 * @author Derek Hulley
 * @since 1.0
 */
public class ZooKeeperClassImporter
{
    private static final Log logger = LogFactory.getLog(ZooKeeperClassImporter.class);
    
    @SuppressWarnings("unused")
    private static final String OPT_ATTEMPTS = "attempts";
    private static final String OPT_ZOOKEEPER_URI = "zkUri";
    private static final String OPT_ZOOKEEPER_PATH = "zkPath";
    @SuppressWarnings("unused")
    private static final String OPT_CLUSTER = "cluster";
    private static final String OPT_HELP = "help";

    /**
     * Dump usage help
     * 
     * @param options           the command line options
     */
    private static void printUsage(Options options)
    {
        HelpFormatter helpFormatter = new HelpFormatter();
        helpFormatter.printHelp(
                "ZooKeeperClassImporter " +
                "-zkUri <ZooKeeper URI> -zkPath <ZooKeeper classes path> ",
                options);
    }
    
    /**
     * Real main method
     * 
     * @param args              command line arguments
     */
    @SuppressWarnings("unused")
    public static void main(String ... args)
    {
        logger.info("Server startup parameters: " + Arrays.asList(args));
        
        Options options = new Options();
        options.addOption(new Option(OPT_ZOOKEEPER_URI, true, "The ZooKeeper server list e.g. '127.0.0.1:3000,127.0.0.1:3001,127.0.0.1:3002'"));
        options.addOption(new Option(OPT_ZOOKEEPER_PATH, true, "The ZooKeeper data path e.g. '/alfresco/bm'"));
        options.addOption(new Option("h", "help", false, "This help message"));
        
        String zkUri = null;
        String zkPath = null;
        try
        {
            CommandLineParser cmdLineParser = new PosixParser();
            CommandLine cmdLine = cmdLineParser.parse(options, args, true);
            if (cmdLine.hasOption(OPT_HELP))
            {
                printUsage(options);
                return;
            }
            zkUri = cmdLine.getOptionValue(OPT_ZOOKEEPER_URI, "127.0.0.1:2181");
            zkPath = cmdLine.getOptionValue(OPT_ZOOKEEPER_PATH, "/alfresco/bm");
        }
        catch (NumberFormatException e)
        {
            printUsage(options);
            return;
        }
        catch (ParseException e)
        {
            printUsage(options);
            return;
        }
    }
}
