/*
 * Copyright (C) 2005-2012 Alfresco Software Limited.
 *
 * This file is part of Alfresco
 *
 * Alfresco is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Alfresco is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with Alfresco. If not, see <http://www.gnu.org/licenses/>.
 */
package org.alfresco.bm.tools;

import java.util.Properties;

import org.alfresco.bm.server.ConfigConstants;
import org.alfresco.http.SharedHttpClientProvider;
import org.springframework.context.support.AbstractApplicationContext;
import org.springframework.context.support.ClassPathXmlApplicationContext;
import org.springframework.core.env.ConfigurableEnvironment;
import org.springframework.core.env.PropertiesPropertySource;

/**
 * Utility code to assist with starting up a test context to aid development by
 * instantiating a test-only context for execution.
 * <p/>
 * Note that, it is a prerequisite that the <b>alfresco-benchmark-server</b> project
 * is imported as a dependency for this test.
 * 
 * @author Derek Hulley
 * @since 1.0
 */
public class BMTestRunner
{
    private final AbstractApplicationContext ctx;
    
    /**
     * Construct the help class that will start and stop the test application context
     * 
     * @param testAppContextLocation        the name of the test application context file.
     *                                      This should be based on the <b>testrun-sample-context.xml</xml>
     * @param threadCount                   the number of threads to run
     * @param mongoUri                      the full MongoDB uri or <tt>null</tt> to use localhost 'test'
     * @param testName                      the name of the test e.g. <b>bm_00005</b>
     * @param testRunName                   the name of the fictitious test run e.g. <b>run_0001</b>
     * @param testProperties                additional properties that will be fed into the test context
     */
    public BMTestRunner(
            String testAppContextLocation,
            int threadCount,
            String mongoUri,
            String testName,
            String testRunName,
            Properties testProperties)
    {
        ctx = new ClassPathXmlApplicationContext(
                new String[] {
                        "server-event-context.xml",
                        "server-mongo-context.xml",
                        testAppContextLocation},
                false);
        // Push in value substitutions
        Properties ctxProperties = new Properties();
        // Server properties
        ctxProperties.put(ConfigConstants.PROP_EVENT_PROCESSOR_THREADS,         "" + threadCount);
        ctxProperties.put(ConfigConstants.PROP_MONGO_URI, (mongoUri == null ?   "mongodb://127.0.0.1:27017/test" : mongoUri));
        ctxProperties.put(ConfigConstants.PROP_MONGO_AUTOCONNECTRETRY,          "true");
        ctxProperties.put(ConfigConstants.PROP_MONGO_CONNECTIONSPERHOST,        "" + (threadCount * 20));
        ctxProperties.put(ConfigConstants.PROP_MONGO_SOCKETTIMEOUT,             "30000");
        ctxProperties.put(ConfigConstants.PROP_MONGO_WRITENUMBER,               "1");
        ctxProperties.put(ConfigConstants.PROP_HTTP_CONNECTION_MAX,             "" + (threadCount * 20));
        ctxProperties.put(ConfigConstants.PROP_HTTP_CONNECTION_TIMEOUT_MS,      "" + SharedHttpClientProvider.DEFAULT_CONNECTION_TIMEOUT_MILLISEC);
        ctxProperties.put(ConfigConstants.PROP_HTTP_SOCKET_TIMEOUT_MS,          "" + SharedHttpClientProvider.DEFAULT_SOCKET_TIMEOUT_MILLISEC);
        ctxProperties.put(ConfigConstants.PROP_HTTP_SOCKET_TTL_MS,              "" + SharedHttpClientProvider.DEFAULT_SOCKET_TTL_MILLISEC);
        ctxProperties.put(ConfigConstants.PROP_CLUSTER, "default");
        ctxProperties.put(ConfigConstants.PROP_SERVER_ID, "test.local");
        ctxProperties.put(ConfigConstants.PROP_TEST_NAME, testName);
        ctxProperties.put(ConfigConstants.PROP_TEST_RUN_NAME, testRunName);
        ctxProperties.put(ConfigConstants.PROP_TEST_RUN_FQN, testName + "." + testRunName);
        // Test properties
        ctxProperties.putAll(testProperties);
        
        ConfigurableEnvironment ctxEnv = ctx.getEnvironment();
        ctxEnv.getPropertySources().addFirst(
                new PropertiesPropertySource(
                        ctxProperties.getProperty(ConfigConstants.PROP_TEST_RUN_FQN),
                        ctxProperties));
        // Ensure clean shutdown when VM goes
        ctx.registerShutdownHook();
        // Now refresh (to load beans) and start
        ctx.refresh();
    }

    /**
     * @return          Returns the application context whether started or not
     */
    public AbstractApplicationContext getCtx()
    {
        return ctx;
    }

    /**
     * Start the application context
     */
    public void start()
    {
        ctx.start();
    }
    
    /**
     * Stop the application context
     */
    public void stop()
    {
        ctx.stop();
        ctx.close();
    }
}
