package org.alfresco.bm.site;

import java.util.Iterator;
import java.util.List;
import java.util.Random;

import org.springframework.beans.factory.InitializingBean;
import org.springframework.data.domain.Sort;
import org.springframework.data.domain.Sort.Direction;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;

import com.mongodb.BasicDBObjectBuilder;
import com.mongodb.DBObject;
import com.mongodb.WriteResult;

/**
 * 
 * Service providing access to {@link SiteData} and {@link SiteMember} storage. All {@link SiteData} and {@link SiteMember} returned from and persisted
 * with this service will be testrun-specific. The testrun-identifier is set in the constructor.
 * 
 * @author steveglover
 * @since 1.3
 */
public class SiteDataServiceImpl implements SiteDataService, InitializingBean
{
    public static interface SiteCallback
    {
        public boolean callback(SiteData site);
    }
    
    private MongoTemplate mongo;
    private String sitesCollectionName;
    private String siteMembersCollectionName;

    public SiteDataServiceImpl(MongoTemplate mongo, String sitesCollectionName, String siteMembersCollectionName)
    {
        this.mongo = mongo;
        this.sitesCollectionName = sitesCollectionName;
        this.siteMembersCollectionName = siteMembersCollectionName;
    }
    
    @Override
    public void afterPropertiesSet() throws Exception
    {
        checkIndexes();
    }
    
    /**
     * Ensure that the MongoDB collection has the required indexes associated with
     * this user bean.
     */
    public void checkIndexes()
    {
        DBObject idxSiteId = BasicDBObjectBuilder
                .start(SiteData.FIELD_SITE_ID, 1)
                .get();
        mongo.getDb().getCollection(sitesCollectionName).ensureIndex(idxSiteId, "idx_SiteId", true);

        DBObject idxSiteMember = BasicDBObjectBuilder
                .start(SiteMember.FIELD_SITE_ID, 1)
                .append(SiteMember.FIELD_USERNAME, 1)
                .get();
        mongo.getDb().getCollection(siteMembersCollectionName).ensureIndex(idxSiteMember, "idx_SiteMember", true);

        DBObject idxSiteCreated = BasicDBObjectBuilder
                .start(SiteData.FIELD_CREATED, 1)
                .append(SiteMember.FIELD_RANDOMIZER, 1)
                .get();
        mongo.getDb().getCollection(sitesCollectionName).ensureIndex(idxSiteCreated, "idx_SiteCreated", false);
        
        DBObject idxSiteMemberCreated = BasicDBObjectBuilder
                .start(SiteMember.FIELD_CREATED, 1)
                .append(SiteMember.FIELD_USERNAME, 1)
                .append(SiteMember.FIELD_RANDOMIZER, 1)
                .get();
        mongo.getDb().getCollection(siteMembersCollectionName).ensureIndex(idxSiteMemberCreated, "idx_SiteMemberCreated", false);
    }
    
    /**
     * Utility method to find a site by siteId
     * 
     */
    @Override
    public SiteData findSiteBySiteId(String siteId)
    {
        Criteria criteriaSiteId = Criteria.where(SiteData.FIELD_SITE_ID).is(siteId);
        Query querySiteById = new Query(criteriaSiteId);
        SiteData site = mongo.findOne(querySiteById, SiteData.class, sitesCollectionName);
        // Done
        return site;
    }
    
    @Override
    public void addSite(SiteData newSite)
    {
        mongo.insert(newSite, sitesCollectionName);
    }

    @Override
    public boolean setSiteCreated(String siteId, boolean created)
    {
        // And save it back to the sites data collection
        Criteria criteriaSiteId = Criteria.where(SiteData.FIELD_SITE_ID).is(siteId);
        Query querySiteBySiteId = new Query(criteriaSiteId);
        Update update = Update.update(SiteData.FIELD_CREATED, created);
        WriteResult result = mongo.updateFirst(querySiteBySiteId, update, sitesCollectionName);
        return (result.getN() > 0);
    }

    @Override
    public void addSiteMember(SiteMember siteMember)
    {
        mongo.insert(siteMember, siteMembersCollectionName);
    }
    
    @Override
    public List<SiteMember> getMembers(String siteId)
    {
        Criteria criteriaSiteMember = Criteria.where(SiteMember.FIELD_SITE_ID).is(siteId);
        Query querySiteMember = new Query(criteriaSiteMember);
        List<SiteMember> members = mongo.find(querySiteMember, SiteMember.class, siteMembersCollectionName);
        return members;
    }
    
    @Override
    public long countSites(String networkId)
    {
        Criteria criteria = Criteria.where(SiteData.FIELD_NETWORKID).is(networkId);
        Query sitesQuery = new Query(criteria);
        long count = mongo.count(sitesQuery, sitesCollectionName);
        // Done
        return count;
    }
    
    @Override
    public long countSiteMembers(String siteId)
    {
        Criteria criteria = Criteria.where(SiteMember.FIELD_SITE_ID).is(siteId);
        Query siteMembersQuery = new Query(criteria);
        long count = mongo.count(siteMembersQuery, siteMembersCollectionName);
        // Done
        return count;
    }
    
    @Override
    public long countSiteMembers()
    {
        long count = mongo.getCollection(siteMembersCollectionName).count();
        // Done
        return count;
    }
    
    @Override
    public long countSites()
    {
        Query sitesQuery = new Query();
        long count = mongo.count(sitesQuery, sitesCollectionName);
        // Done
        return count;
    }
    
    @Override
    public long countSites(boolean created)
    {
        long count = 0L;
        if (created)
        {
            Criteria sitesCriteria = Criteria.where(SiteData.FIELD_CREATED).is(Boolean.valueOf(created));
            Query sitesQuery = new Query(sitesCriteria);
            count = mongo.count(sitesQuery, sitesCollectionName);
        }
        else
        {
            Query sitesQuery = new Query();
            count = mongo.count(sitesQuery, sitesCollectionName);
        }
        // Done
        return count;
    }
    
    @Override
    public Sites getSites(int max)
    {
        return new Sites(max);
    }

    // Mark the site as having its members created
    public void markSiteMembersCreated(SiteData site)
    {
        Criteria criteriaSite = Criteria.where(SiteData.FIELD_SITE_ID).is(site.getSiteId());
        Query querySite = new Query(criteriaSite);
        Update update = Update.update(SiteData.FIELD_HAS_MEMBERS, Boolean.TRUE);
        mongo.updateFirst(querySite, update, sitesCollectionName);
    }

    private Integer getMaxSiteRandomizer()
    {
    	Query query = new Query();
    	query.with(new Sort(Direction.DESC, SiteData.FIELD_RANDOMIZER));
    	query.limit(1);

    	SiteData firstResult = (SiteData) mongo.findOne(query, SiteData.class, sitesCollectionName);
    	return firstResult == null ? null : firstResult.getRandomizer();
    }

    private Integer getMinSiteRandomizer()
    {
    	Query query = new Query();
        query.with(new Sort(Direction.ASC, SiteData.FIELD_RANDOMIZER));
    	query.limit(1);

    	SiteData firstResult = (SiteData) mongo.findOne(query, SiteData.class, sitesCollectionName);
    	return (firstResult == null ? null : firstResult.getRandomizer());
    }
    
    private Integer getMaxSiteMemberRandomizer()
    {
    	Query query = new Query();
        query.with(new Sort(Direction.DESC, SiteData.FIELD_RANDOMIZER));
    	query.limit(1);

    	SiteMember firstResult = (SiteMember) mongo.findOne(query, SiteMember.class, siteMembersCollectionName);
    	return firstResult == null ? null : firstResult.getRandomizer();
    }

    private Integer getMinSiteMemberRandomizer()
    {
    	Query query = new Query();
        query.with(new Sort(Direction.ASC, SiteData.FIELD_RANDOMIZER));
    	query.limit(1);

    	SiteMember firstResult = (SiteMember) mongo.findOne(query, SiteMember.class, siteMembersCollectionName);
    	return (firstResult == null ? null : firstResult.getRandomizer());
    }
    
    @Override
    public SiteData randomSite(String networkId)
    {
    	Integer max = getMaxSiteRandomizer();
    	Integer min = getMinSiteRandomizer();
    	if(min != null && max != null)
    	{
	        int r = (int)(Math.random() * (max - min));

    		Criteria criteria = Criteria.where(SiteData.FIELD_CREATED).is(true)
    				.and(SiteData.FIELD_RANDOMIZER).gte(r);
	        Query query = new Query(criteria);
	        SiteData site = mongo.findOne(query, SiteData.class, sitesCollectionName);
	        if(site == null)
	        {
	        	criteria = Criteria.where(SiteData.FIELD_CREATED).is(true)
	        			.and(SiteData.FIELD_RANDOMIZER).lte(r);
	            query = new Query(criteria);
	            site = mongo.findOne(query, SiteData.class, sitesCollectionName);
	        }

	        return site;
    	}
    	else
    	{
    		return null;
    	}
    }

    @Override
    public SiteMember randomMember(String siteId)
    {
    	Integer max = getMaxSiteMemberRandomizer();
    	Integer min = getMinSiteMemberRandomizer();
    	if(min != null && max != null)
    	{
	        int r = (int)(Math.random() * (max - min));

	        Criteria criteria = Criteria.where(SiteMember.FIELD_CREATED).is(true)
	        		.and(SiteMember.FIELD_SITE_ID).is(siteId)
	        		.and(SiteMember.FIELD_RANDOMIZER).gte(r);
	        Query query = new Query(criteria);
	        SiteMember siteMember = mongo.findOne(query, SiteMember.class, sitesCollectionName);
	        if(siteMember == null)
	        {
		        criteria = Criteria.where(SiteMember.FIELD_CREATED).is(true)
		        		.and(SiteMember.FIELD_SITE_ID).is(siteId)
		        		.and(SiteMember.FIELD_RANDOMIZER).lte(r);
	            query = new Query(criteria);
	            siteMember = mongo.findOne(query, SiteMember.class, sitesCollectionName);
	        }

	        return siteMember;
    	}
    	else
    	{
    		return null;
    	}
    }
    
    @Override
    public List<SiteData> getSitesPendingCreation(int startIndex, int count)
    {
        return getSites(false, startIndex, count);
    }
    
    @Override
    public Iterator<SiteData> sitesIterator(String networkId, boolean created)
    {
        Criteria sitesCriteria = Criteria.where(SiteData.FIELD_CREATED).is(Boolean.valueOf(created)).and(SiteData.FIELD_NETWORKID).is(networkId);
        Query sitesQuery = new Query(sitesCriteria);
        Iterator<SiteData> sitesIt = mongo.find(sitesQuery, SiteData.class, sitesCollectionName).iterator();
        return sitesIt;
    }
    
    @Override
    public boolean isSiteMember(String siteId, String userId)
    {
        SiteMember siteMember = getSiteMember(siteId, userId);
        return siteMember != null;
    }
    
    @Override
    public SiteMember getSiteMember(String siteId, String userId)
    {
        Criteria siteMemberCriteria = Criteria.where(SiteMember.FIELD_SITE_ID).is(siteId).and(SiteMember.FIELD_USERNAME).is(userId);
        Query siteMemberQuery = new Query(siteMemberCriteria);
        SiteMember siteMember = mongo.findOne(siteMemberQuery, SiteMember.class, siteMembersCollectionName);
        return siteMember;
    }

    @Override
    public void setSiteMemberCreated(String siteId, String userId, SiteRole role, boolean created)
    {
        SiteMember siteMember = getSiteMember(siteId, userId);
        siteMember.setCreated(Boolean.TRUE);

        // And save it back to the site members data collection
        Criteria criteriaSiteMember = Criteria.where(SiteMember.FIELD_SITE_ID).is(siteId).and(SiteMember.FIELD_USERNAME).is(userId);
        Query querySiteMember = new Query(criteriaSiteMember);
        Update update = Update.update(SiteMember.FIELD_CREATED, created);
        mongo.updateFirst(querySiteMember, update, siteMembersCollectionName);
    }

    @Override
    public List<SiteMember> getSiteMembersPendingCreation(int startIndex, int count)
    {
        return getSiteMembersPendingCreation(false, startIndex, count);
    }
    
    protected List<SiteData> getSites(boolean created, int startIndex, int count)
    {
        Criteria usersCriteria = Criteria.where(SiteData.FIELD_CREATED).is(Boolean.valueOf(created));
        Query sitesQuery = new Query(usersCriteria);
        sitesQuery.with(new Sort(Direction.ASC, SiteData.FIELD_RANDOMIZER));
        sitesQuery.skip(startIndex).limit(count);

        List<SiteData> sites = mongo.find(sitesQuery, SiteData.class, sitesCollectionName);
        
        // Done
        return sites;
    }
    
    protected List<SiteMember> getSiteMembersPendingCreation(boolean created, int startIndex, int count)
    {
        Criteria siteMembersCriteria = Criteria.where(SiteMember.FIELD_CREATED).is(Boolean.valueOf(created));
        Query siteMembersQuery = new Query(siteMembersCriteria);
        siteMembersQuery.with(new Sort(Direction.ASC, SiteData.FIELD_RANDOMIZER));
        siteMembersQuery.skip(startIndex).limit(count);

        List<SiteMember> siteMembers = mongo.find(siteMembersQuery, SiteMember.class, siteMembersCollectionName);

        // Done
        return siteMembers;
    }
    
    /**
     * A collection of sites.
     * 
     * @author steveglover
     *
     */
    public class Sites
    {
        private int max;
        private Random random = new Random();
        
        public Sites(int max)
        {
            this.max = max;
        }

         public void forEach(SiteCallback callback)
        {
            long numSites = countSites();
            
            int skip = random.nextInt((int)(numSites - max));
            Query query = new Query().skip(skip).limit(max);
            List<SiteData> sites = mongo.find(query, SiteData.class, sitesCollectionName);
            for(SiteData site : sites)
            {
                callback.callback(site);
            }
        }
    }
}
