package org.alfresco.bm.site;

import java.util.Iterator;
import java.util.List;

import org.alfresco.bm.site.SiteDataServiceImpl.Sites;

/**
 * Service interface providing mirror data for Alfresco sites
 * 
 * @author Steven Glover
 * @since  1.3
 */
public interface SiteDataService
{
    public interface SiteCallback
    {
        public boolean callback(SiteData site);
    }

    /**
     * Utility method to find a site by siteId
     */
    SiteData findSiteBySiteId(String siteId);
    
    /**
     * Add a site
     */
    void addSite(SiteData newSite);
    
    /**
     * Mark the site as created or not created
     */
    boolean setSiteCreated(String siteId, boolean created);
    
    /**
     * Count the number of sites that have been created, or are yet to be created
     * 
     * @param created
     * @return the number of sites matching the "created" criterion
     */
    long countSites(boolean created);
    
    /**
     * Count the number of sites in the given network.
     * 
     * @param networkId
     * @return the number of sites in the given network
     */
    long countSites(String networkId);
    
    /**
     * Count the number of site members of the given site.
     * @param siteId
     * @return the number of site members of the given site
     */
    long countSiteMembers(String siteId);
    
    /**
     * Count the total number of site members.
     * @return the total number of site members
     */
    long countSiteMembers();
    
    /**
     * Count the number of sites
     * 
     * @return the total number of sites
     */
    long countSites();
    
    /**
     * Get a list of sites up to a maximum size of "max"
     * 
     * @return a collection of max sites
     */
    Sites getSites(int max);
    
    /**
     * Get a random site in the network given by "networkId"
     * 
     * Note that this is useful only for large numbers of sites.
     * 
     * @return the siteId of a random site in the given network
     */
    SiteData randomSite(String networkId);
    
    /**
     * Get sites pending creation, up to a maximum of "count"
     * 
     * @return a list of sites pending creation
     */
    List<SiteData> getSitesPendingCreation(int startIndex, int count);
    
    /**
     * Iterate through sites in the network "networkId" that have been created/not created.
     * 
     * @param networkId
     * @param created
     * @return an iterator over sites in the given network that match the created flag
     */
    Iterator<SiteData> sitesIterator(String networkId, boolean created);

    /**
     * Add a site member, initially not created.
     */
    void addSiteMember(SiteMember siteMember);
    
    /**
     * Get members of site "siteId"
     * 
     * @return a list of members of the site
     */
    List<SiteMember> getMembers(String siteId);
    
    /**
     * Get a random site member in the site given by "siteId"
     * 
     * Note that this is useful only for large numbers of sites.
     * 
     * @return a random member of the given site
     */
    SiteMember randomMember(String siteId);
    
    /**
     * Mark all members of the given site as having been created.
     * 
     * @param site
     */
    void markSiteMembersCreated(SiteData site);

    /**
     * Get a specific site member
     * 
     * @param siteId
     * @param userId
     * @return get a specific site member
     */
    public SiteMember getSiteMember(String siteId, String userId);

    /**
     * Get a list of site members pending creation, starting at index "startIndex" up to a maximum size of "count"
     * 
     * @return a list of site members pending creation
     */
    List<SiteMember> getSiteMembersPendingCreation(int startIndex, int count);
    
    /**
     * Is userId a member of the site "siteId"?
     * 
     * @param siteId site id
     * @param userId site member id
     * @return true if the user is a member of the site, false otherwise
     */
    boolean isSiteMember(String siteId, String userId);
    
    /**
     * Mark the site member as created or not created
     */
    void setSiteMemberCreated(String siteId, String userId, SiteRole role, boolean created);
}
