package org.alfresco.bm.site;

import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import org.alfresco.bm.Services;
import org.alfresco.bm.event.AbstractEventProcessor;
import org.alfresco.bm.event.Event;
import org.alfresco.bm.event.EventResult;
import org.alfresco.bm.user.UserData;
import org.alfresco.bm.user.UserDataService;

/**
 * Prepares sites for creation by populating the sites collection.
 * 
 * The number of sites is driven by:
 * 
 * maxNetworks: the maximum number of networks to include. -1 indicates just use the networks implicit in the users collection.
 * maxSitesPerNetwork: the maximum number of sites per network. -1 indicates just use the sites defined in the sites collection.
 * startSite: start number site names from here.
 * 
 * @author steveglover
 *
 */
public class PrepareSites extends AbstractEventProcessor
{
    public static final String EVENT_NAME_SITES_PREPARED = "sitesPrepared";
    public static final int DEFAULT_MAX_NETWORKS = Integer.MAX_VALUE; // all
    public static final int DEFAULT_MAX_SITES_PER_NETWORK = Integer.MAX_VALUE; // all

    private int maxNetworks;
    private int maxSitesPerNetwork;
    private String eventNameSitesPrepared;
    private UserDataService userDataService;
    private SiteDataService siteDataService;

    private int startSite = 0;

    /**
     * @param services              data collections
     */
    public PrepareSites(Services services)
    {
        super();
        this.userDataService = services.getUserDataService();
        this.siteDataService = services.getSiteDataService(); 
        this.maxSitesPerNetwork = DEFAULT_MAX_SITES_PER_NETWORK;
        this.maxNetworks = DEFAULT_MAX_NETWORKS;
        this.eventNameSitesPrepared = EVENT_NAME_SITES_PREPARED;
    }
    
    public void init()
    {
    }

    public void setMaxNetworks(int maxNetworks)
    {
        this.maxNetworks = maxNetworks;
    }

    public void setMaxSitesPerNetwork(int maxSitesPerNetwork)
    {
        this.maxSitesPerNetwork = maxSitesPerNetwork;
    }

    public void setStartSite(int startSite)
    {
        this.startSite = startSite;
    }

    /**
     * Override the {@link #EVENT_NAME_SITES_PREPARED default} event name when sites have been created.
     */
    public void setEventNameSitesPrepared(String eventNameSitesPrepared)
    {
        this.eventNameSitesPrepared = eventNameSitesPrepared;
    }

    @SuppressWarnings("rawtypes")
    @Override
    public EventResult processEvent(Event event) throws Exception
    {
        int sitesCount = 0;
        int networksCount = 0;

        Iterator networksIt = userDataService.networksIterator();
        while(networksIt.hasNext() && networksCount < maxNetworks)
        {
            final String networkId = (String)networksIt.next();

            List<UserData> usersInNetwork = userDataService.getUsersInDomain(networkId, 0, maxSitesPerNetwork);
            int userIndex = 0;

            for (int j = startSite; j < maxSitesPerNetwork + startSite; j++)
            {
                String siteId = String.format("%s.%05d", networkId, j);
                SiteData site = siteDataService.findSiteBySiteId(siteId);
                if(site != null)
                {
                    // Site already exists
                    continue;
                }

                if(userIndex >= usersInNetwork.size())
                {
                    userIndex = 0;
                }
                UserData creator = usersInNetwork.get(userIndex);
                
                // Create data
                final SiteData newSite = new SiteData();
                newSite.setDescription("");
                newSite.setSiteId(siteId);
                newSite.setSitePreset("preset");
                newSite.setTitle(siteId);
                newSite.setVisibility(SiteVisibility.getRandomVisibility());
                newSite.setType("st:site");
                newSite.setNetworkId(networkId);
                newSite.setCreatedBy(creator.getEmail());
                newSite.setCreated(Boolean.FALSE);

                // Persist
                siteDataService.addSite(newSite);
                sitesCount++;
                userIndex++;
            }

            networksCount++;
        }

        // We need an event to mark completion
        String msg = "Prepared " + sitesCount + " sites in " + networksCount + " networks.";
        Event outputEvent = new Event(eventNameSitesPrepared, 0L, msg);

        // Create result
        EventResult result = new EventResult(msg, Collections.singletonList(outputEvent));

        // Done
        if (logger.isDebugEnabled())
        {
            logger.debug(msg);
        }
        return result;
    }
}
