package org.alfresco.bm.site;

import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import org.alfresco.bm.Services;
import org.alfresco.bm.event.AbstractEventProcessor;
import org.alfresco.bm.event.Event;
import org.alfresco.bm.event.EventResult;
import org.alfresco.bm.user.UserData;
import org.alfresco.bm.user.UserDataService;
import org.springframework.social.alfresco.api.entities.Role;

/**
 * Prepares site members for creation by populating the site members collection.
 * 
 * The number of site members to create is driven by:
 * 
 * maxNetworks: the maximum number of networks to include. -1 indicates just use the networks implicit in the users collection.
 * maxSites: in each network, the maximum number of sites to include. -1 indicates just use all the sites in the sites collection.
 * maxMembersPerSite: the maximum number of members per site (currently from the same network only).
 *    -1 indicates just use all the users in the users collection for the current network.
 * 
 * TODO Currently, site memberships only for users in the same network. Expand to include other networks.
 * 
 * @author steveglover
 *
 */
public class PrepareSiteMembers extends AbstractEventProcessor
{
    public static final String EVENT_NAME_SITE_MEMBERS_PREPARED = "siteMembersPrepared";

    public static final int DEFAULT_MAX_NETWORKS = Integer.MAX_VALUE; // all
    public static final int DEFAULT_MAX_SITES = Integer.MAX_VALUE; // all
    public static final int DEFAULT_MAX_MEMBERS_PER_SITE = Integer.MAX_VALUE; // all
            
    private String eventNameSiteMembersPrepared;
    private UserDataService userDataService;
    private SiteDataService siteDataService;

    private int maxNetworks;
    private int maxSites;
    private int maxMembersPerSite;
    
    /**
     * @param services              data collections
     */
    public PrepareSiteMembers(Services services)
    {
        super();
        this.userDataService = services.getUserDataService();
        this.siteDataService = services.getSiteDataService(); 
        this.eventNameSiteMembersPrepared = EVENT_NAME_SITE_MEMBERS_PREPARED;
        this.maxNetworks = DEFAULT_MAX_NETWORKS;
        this.maxSites = DEFAULT_MAX_SITES;
        this.maxMembersPerSite = DEFAULT_MAX_MEMBERS_PER_SITE;
    }
    
    public void setMaxNetworks(int maxNetworks)
    {
        this.maxNetworks = maxNetworks == -1 ? Integer.MAX_VALUE : maxNetworks;
    }

    public void setMaxSites(int maxSites)
    {
        this.maxSites = maxSites == -1 ? Integer.MAX_VALUE : maxSites;
    }

    public void setMaxMembersPerSite(int maxMembersPerSite)
    {
        this.maxMembersPerSite = maxMembersPerSite == -1 ? Integer.MAX_VALUE : maxMembersPerSite;
    }

    /**
     * Override the {@link #EVENT_NAME_SITE_MEMBERS_PREPARED default} event name when site members have been created.
     */
    public void setEventNameSiteMembersPrepared(String eventNameSiteMembersPrepared)
    {
        this.eventNameSiteMembersPrepared = eventNameSiteMembersPrepared;
    }

    @Override
    public EventResult processEvent(Event event) throws Exception
    {
        int sitesCount = 0;
        int networksCount = 0;
        int membersCount = 0;

        long expectedTotal = maxNetworks * maxSites * maxMembersPerSite;
        long numSiteMembers = siteDataService.countSiteMembers();
        if(numSiteMembers < expectedTotal)
        {
	        Iterator<String> networksIt = userDataService.networksIterator();
	        while(networksIt.hasNext() && networksCount++ < maxNetworks)
	        {
	            String networkId = networksIt.next();

	            Iterator<SiteData> sitesIt = siteDataService.sitesIterator(networkId, true);
	            while(sitesIt.hasNext() && sitesCount++ < maxSites)
	            {
	                final SiteData site = (SiteData)sitesIt.next();

	                List<UserData> users = userDataService.getUsersInDomain(networkId, 0, maxMembersPerSite);
	                for(UserData user : users)
	                {
	                    String userId = user.getEmail();
	                    Role role = Role.SiteContributor; // TODO more random, spread of roles?
	                    if(!siteDataService.isSiteMember(site.getSiteId(), userId))
	                    {
	                        SiteMember siteMember = new SiteMember(userId, site.getSiteId(), SiteRole.valueOf(role.toString()));
	                        siteDataService.addSiteMember(siteMember);
	                        membersCount++;
	                    }                        
	                }
	            }
	        }
        }

        // We need an event to mark completion
        String msg = "Created " + membersCount + " site members in " + sitesCount + " sites in " + networksCount + " networks.";
        Event outputEvent = new Event(eventNameSiteMembersPrepared, 0L, msg);

        // Create result
        EventResult result = new EventResult(msg, Collections.singletonList(outputEvent));

        // Done
        if (logger.isDebugEnabled())
        {
            logger.debug(msg);
        }
        return result;
    }
}
