/*
 * Copyright (C) 2005-2012 Alfresco Software Limited.
 *
 * This file is part of Alfresco
 *
 * Alfresco is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Alfresco is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with Alfresco. If not, see <http://www.gnu.org/licenses/>.
 */
package org.alfresco.bm.server;

import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.RejectedExecutionException;

import org.alfresco.bm.event.DoNothingEventProcessor;
import org.alfresco.bm.event.Event;
import org.alfresco.bm.event.EventProcessor;
import org.alfresco.bm.event.EventProcessorRegistry;
import org.alfresco.bm.event.EventRecord;
import org.alfresco.bm.event.EventService;
import org.alfresco.bm.event.EventWork;
import org.alfresco.bm.event.ResultService;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.BeansException;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.ApplicationListener;
import org.springframework.context.event.ApplicationContextEvent;
import org.springframework.context.event.ContextStartedEvent;
import org.springframework.context.event.ContextStoppedEvent;

/**
 * A <i>master</i> controlling thread that ensures that:
 * <ul>
 *  <li>reads events from the queue</li>
 *  <li>checks out threads to process events</li>
 *  <li>monitors event processors</li>
 *  <li>records event executions</li>
 *  <li>handles exceptions e.g. events that take too long to process</li>
 * </ul>
 * Application events are monitored to start, stop and otherwise control
 * the threads.
 * 
 * @author Derek Hulley
 * @since 1.0
 */
public class EventController
        extends Thread
        implements
            ApplicationListener<ApplicationContextEvent>,
            ApplicationContextAware,
            ConfigConstants
{
    private static final long DEFAULT_QUEUE_POLL_TIME_MS = 1000L;
    private static final long DEFAULT_QUEUE_REST_AGE_MS = 1000L;
    
    private static final Log logger = LogFactory.getLog(EventController.class);
    
    private final String serverId;
    private final String testRunFqn;
    private final EventService eventService;
    private final EventProcessorRegistry eventProcessors;
    private final ExecutorService executor;
    private final ResultService resultService;

    private long queuePollTimeMs = DEFAULT_QUEUE_POLL_TIME_MS;
    private long queueRestAgeMs = DEFAULT_QUEUE_REST_AGE_MS;

    private ApplicationContext ctx;
    private RunState runState;
    private EventProcessor doNothingProcessor = new DoNothingEventProcessor();
    
    /**
     * Construct the controller
     * 
     * @param serverId          the server controlling the events
     * @param testRunFqn        the fully qualified name of the test run
     * @param eventService      the source of events that will be pushed for execution
     * @param eventProcessors   the registry of processors for events
     * @param executor          the service that will provide the worker threads for processing
     * @param resultService     the service used to store and retrieve event results
     */
    public EventController(
            String serverId,
            String testRunFqn,
            EventService eventService,
            EventProcessorRegistry eventProcessors,
            ExecutorService executor,
            ResultService resultService)
    {
        super(new ThreadGroup("EventProcessing"), "ProcessorController." + testRunFqn);
        super.setDaemon(false);         // Requires explicit shutdown
        
        this.serverId = serverId;
        this.testRunFqn = testRunFqn;
        this.eventService = eventService;
        this.eventProcessors = eventProcessors;
        this.executor = executor;
        this.resultService = resultService;
        
        this.runState = RunState.RUN;
    }

    /**
     * The time between polls of the event provider.
     * Default is {@link #DEFAULT_QUEUE_POLL_TIME_MS}.
     * 
     * @param queuePollTimeMs       the time between checks for events
     */
    public void setQueuePollTimeMs(long queuePollTimeMs)
    {
        this.queuePollTimeMs = queuePollTimeMs;
    }

    /**
     * Set the time allowed before events can be requeued.
     * Default is {@link #DEFAULT_QUEUE_REST_AGE_MS}.
     */
    public void setQueueRestAgeMs(long queueRestAgeMs)
    {
        this.queueRestAgeMs = queueRestAgeMs;
    }

    /**
     * Record the application context for shutdown once processing has finished
     */
    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException
    {
        this.ctx = applicationContext;
    }

    /**
     * The application context is started externally but may be stopped internally
     */
    @Override
    public synchronized void onApplicationEvent(ApplicationContextEvent event)
    {
        if (event instanceof ContextStartedEvent)
        {
            this.start();
        }
        else if (event instanceof ContextStoppedEvent)
        {
            this.runState = RunState.STOP;
            // If someone else is making this call then make sure we wait for the thread to kill itself
            if (!Thread.currentThread().equals(this))
            {
                // Wake the thread up
                this.notify();
                // Wait for the thread to stop
                try { this.join(); } catch (InterruptedException e) {}
            }
        }
    }
    
    @Override
    public synchronized void run()
    {
        try
        {
            runImpl();
        }
        catch (Throwable e)
        {
            // We can ignore errors if we have been told to stop
            if (runState == RunState.RUN)
            {
                logger.error("\tEvent processing terminated with error: " + testRunFqn, e);
            }
            else
            {
                logger.debug("\tEvent processing terminated with error: " + testRunFqn, e);
            }
        }
    }
    
    /**
     * Do the actual run but without concern for exceptions, which will be logged.
     */
    private synchronized void runImpl()
    {
        logger.info("\tEvent processing started: " + testRunFqn);

        boolean foundEvent = true;              // Don't wait long if we are finding events
        while (runState == RunState.RUN)
        {
            // Only do a wait if we didn't manage to find anything interesting
            if (foundEvent)
            {
                Thread.yield();
            }
            else
            {
                try { this.wait(queuePollTimeMs); } catch (InterruptedException e) {}
            }
            // Grab an event
            foundEvent = false;
            long now = System.currentTimeMillis();
            Event event = eventService.nextEvent(serverId, now, (now - queueRestAgeMs));
            // Do we have an event to process?
            if (event == null)
            {
                long count = eventService.count();
                if (count == 0)
                {
                    // Look in the results to see if the run was started at some point
                    List<EventRecord> startRecords = resultService.getResults(Event.EVENT_NAME_START, 0, 1);
                    if (startRecords.size() == 0)
                    {
                        // The test has not *ever* been started.
                        // We do that now
                        event = new Event(Event.EVENT_NAME_START, 0L, null);
                        eventService.putEvent(event);
                        foundEvent = true;
                    }
                    else
                    {
                        // The test was started but there are no more events remaining.
                        // Quit
                        if (ctx != null)        // The controller might have been run manually
                        {
                            ctx.publishEvent(new ContextStoppedEvent(ctx));
                        }
                    }
                }
                // Go back to the queue
                continue;
            }
            foundEvent = true;
            // We have an event
            String eventName = event.getName();
            // Have we been told to stop the test?
            if (Event.EVENT_NAME_STOP.equals(eventName))
            {
                // Shutdown the context
                runState = RunState.STOP;
                continue;
            }
                
            // Find the processor for the event
            EventProcessor processor = getProcessor(event);
            // Schedule it
            EventWork work = new EventWork(
                    serverId, testRunFqn,
                    event,
                    processor, eventService, resultService);
            try
            {
                executor.execute(work);
            }
            catch (RejectedExecutionException e)
            {
                // Failed to put the event on the queue (it is probably full)
                foundEvent = false;
            }
        }
        
        logger.info("\tEvent processing stopped: " + testRunFqn);
    }
    
    /**
     * Get a processor for the event.  If an event is not mapped, and error is logged
     * and the event is effecitvely absorbed.
     */
    private EventProcessor getProcessor(Event event)
    {
        String eventName = event.getName();
        EventProcessor processor = eventProcessors.getProcessor(eventName);
        if (processor == null)
        {
            if (logger.isDebugEnabled())
            {
                logger.debug("\n" +
                        "No event processor mapped to event: \n" +
                        "   Event name: " + eventName + "\n" +
                        "   Event:      " + event);
            }
            processor = doNothingProcessor;
        }
        return processor;
    }
}
