/*
 * Copyright (C) 2005-2012 Alfresco Software Limited.
 *
 * This file is part of Alfresco
 *
 * Alfresco is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Alfresco is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with Alfresco. If not, see <http://www.gnu.org/licenses/>.
 */
package org.alfresco.bm.server;

import java.io.File;
import java.io.FileWriter;
import java.util.Properties;

import org.I0Itec.zkclient.exception.ZkException;
import org.alfresco.config.ConfigChildListener;
import org.alfresco.config.ConfigClassesListener;
import org.alfresco.config.ConfigDataListener;
import org.alfresco.config.ConfigException;
import org.alfresco.config.ConfigService;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.NoSuchBeanDefinitionException;
import org.springframework.context.support.AbstractApplicationContext;
import org.springframework.context.support.ClassPathXmlApplicationContext;
import org.springframework.context.support.FileSystemXmlApplicationContext;
import org.springframework.core.env.ConfigurableEnvironment;
import org.springframework.core.env.PropertiesPropertySource;

/**
 * Encapsulation of data and behaviour associated with a specific test run.
 * <p/>
 * TODO: Need to log errors to distributed sink 
 * 
 * @author Derek Hulley
 * @since 1.0
 */
public class BMTestRun implements ConfigDataListener, ConfigClassesListener, ConfigChildListener, ConfigConstants
{
    private static final Log logger = LogFactory.getLog(BMTestRun.class);
    
    //Enforce application context to use absolute path
    private static final String FILE_PREFIX = "file:";
    
    /**
     * The key that represents the unique data defining a specific test run
     * i.e. the {@link BMTestRun#testName} and {@link BMTestRun#testRunName}.
     * 
     * @author Derek Hulley
     * @since 1.0
     */
    public static class Key implements Comparable<Key>
    {
        private final String testName;
        private final String testRunName;
        public Key(String testName, String testRunName)
        {
            this.testName = testName;
            this.testRunName = testRunName;
        }
        @Override
        public String toString()
        {
            return "Key [testName=" + testName + ", testRunName=" + testRunName + "]";
        }
        @Override
        public int hashCode()
        {
            return testName.hashCode() + 31 * testRunName.hashCode();
        }
        @Override
        public boolean equals(Object obj)
        {
            if (obj == null || !(obj instanceof Key))
            {
                return false;
            }
            Key that = (Key) obj;
            return this.testName.equals(that.testName) && this.testRunName.equals(that.testRunName);
        }
        @Override
        public int compareTo(Key that)
        {
            int testNameCompare = this.testName.compareTo(that.testName);
            return ((testNameCompare == 0) ? (this.testRunName.compareTo(that.testRunName)) : testNameCompare);
        }
    }

    // Constructor-provided data
    private final BMServer server;
    private final String cluster;
    private final String testName;
    private final String testRunName;
    private final String testRunFqn; 
    // State data
    private AbstractApplicationContext testRunCtx;
    /** The desired run state of the server. */
    private RunState desiredRunState;

    /**
     * Consructor containing enough data for the object to go off and start itself
     * <p/>
     * Initially, values are retrieved without data watches
     * 
     * @param server            the server constructing the test run
     * @param testRunKey        the unique test run key
     * @param configService      the configuration loader
     */
    public BMTestRun(BMServer server, Key testRunKey)
    {
        this.server = server;
        this.cluster = server.getCluster();
        this.testName = testRunKey.testName;
        this.testRunName = testRunKey.testRunName;
        this.testRunFqn = (server.getCluster() + "." + testName + "." + testRunName);
        
        this.desiredRunState = RunState.RUN;
        
        if (server.getApplicationContext() == null)
        {
            throw new IllegalStateException("A test run can only be constructed by a running server.");
        }
        
        // Check the naming conventions
        if (!testRunFqn.matches("^[a-zA-Z0-9_\\.]*$"))
        {
            throw new ConfigException("\n" +
                    "Configuration paths may only contain letters (mixed case), numbers, period ('.') and underscore ('_').\n" +
                    "   Test run path is invalid: " + testRunFqn);
        }
    }
    
    @Override
    public String toString()
    {
        return testRunFqn;
    }
    
    /**
     * Equality is based on the test and run names.  This must be unique within the server.
     */
    @Override
    public boolean equals(Object obj)
    {
        if (this == obj) return true;
        if (obj == null) return false;
        if (getClass() != obj.getClass()) return false;
        BMTestRun other = (BMTestRun) obj;
        if (!testName.equals(other.testName)) return false;
        if (!testRunName.equals(other.testRunName)) return false;
        return true;
    }

    /**
     * 
     * @return          Returns the unique key for the test run
     */
    public Key getKey()
    {
        return new Key(testName, testRunName);
    }

    /**
     * Forces a check for {@link RunState#RESTART restart} requests.
     */
    public synchronized void checkForRestart()
    {
        if (desiredRunState == RunState.RESTART)
        {
            try
            {
                logger.info("Reloading test run '" + this + "'.");
                reload();
            }
            catch (Throwable e)
            {
                logger.error("Unable to reload test run '" + this + "'.", e);
            }
        }
        // Reset the desired run state
        desiredRunState = RunState.RUN;
    }
    
    /**
     * Explicit call to start the test run application.  No exceptions are thrown.
     */
    public synchronized void start()
    {
        logger.info("Loading test run '" + testRunFqn + "'.");
        try
        {
            // Register for events
            reload();
        }
        catch (Throwable e)
        {
            logger.error("Unable to load test run '" + this + "'.", e);
        }
    }

    /**
     * Explicit call to stop the test run application
     */
    public synchronized void stop()
    {
        logger.info("Stopping test run: " + testRunFqn);
        if (testRunCtx != null)
        {
            // Get the controller thread
            EventController controller = null;
            try
            {
                // This bean might not be present if the test context was not started
                controller = (EventController) testRunCtx.getBean("eventController");
            }
            catch (NoSuchBeanDefinitionException e)
            {
                // Quite possible
            }
            // Stop the context
            testRunCtx.stop();
            if (controller != null)
            {
                // Wait for the thread to kill itself
                try { controller.join(5000L); } catch (InterruptedException e) {}
            }
            // Close down completely
            testRunCtx.close();
            testRunCtx = null;
        }
    }
    
    @Override
    public synchronized void classesChanged(String path)
    {
        this.desiredRunState = RunState.RESTART;
    }

    @Override
    public synchronized void dataChanged(String path, boolean deleted)
    {
        this.desiredRunState = RunState.RESTART;
    }
    
    @Override
    public void childrenChanged(String path)
    {
        this.desiredRunState = RunState.RESTART;
    }

    /**
     * @see BMTestRun#createTestStructure(ConfigService, String, String)
     */
    private synchronized void createTestStructure(ConfigService configService)
    {
        createTestStructure(configService, cluster, testName, testRunName);
    }
    
    /**
     * Creates necessary configuration nodes, avoiding any concurrency conditions when running
     * multiple servers in the same cluster configuration.
     * <p/>
     * Note that <b>./PATH_TESTS/testName/PATH_CLASSES</b> is not created by default.
     * 
     * @param configService             the configuration service rooted in the correct location
     */
    public static void createTestStructure(ConfigService configService, String cluster, String testName, String testRunName)
    {
        Properties emptyProperties = new Properties();
        
        Properties runStateStop = new Properties();
        runStateStop.setProperty(PROP_CONTROL_RUN_STATE, DEFAULT_RUNSTATE);
        
        if (!configService.exists(PATH_CLUSTERS, cluster, PATH_TESTS))
        {
            try
            {
                configService.setString(
                        null, false, false, null,
                        PATH_CLUSTERS, cluster, PATH_TESTS);
            }
            catch (ZkException e)
            {
                // Ignore
            }
        }
        if (!configService.exists(PATH_CLUSTERS, cluster, PATH_TESTS, testName))
        {
            try
            {
                configService.setString(
                        null, false, false, null,
                        PATH_CLUSTERS, cluster, PATH_TESTS, testName);
            }
            catch (ZkException e)
            {
                // Ignore
            }
        }
        if (!configService.exists(PATH_CLUSTERS, cluster, PATH_TESTS, testName, PATH_CONFIG))
        {
            try
            {
                configService.setString(
                        null, false, false, null,
                        PATH_CLUSTERS, cluster, PATH_TESTS, testName, PATH_CONFIG);
            }
            catch (ZkException e)
            {
                // Ignore
            }
        }
        if (!configService.exists(PATH_CLUSTERS, cluster, PATH_TESTS, testName, PATH_JARS))
        {
            try
            {
                configService.setString(
                        null, false, false, null,
                        PATH_CLUSTERS, cluster, PATH_TESTS, testName, PATH_JARS);
            }
            catch (ZkException e)
            {
                // Ignore
            }
        }
        if (!configService.exists(PATH_CLUSTERS, cluster, PATH_TESTS, testName, PATH_RUNS))
        {
            try
            {
                configService.setString(
                        null, false, false, null,
                        PATH_CLUSTERS, cluster, PATH_TESTS, testName, PATH_RUNS);
            }
            catch (ZkException e)
            {
                // Ignore
            }
        }
        if (!configService.exists(PATH_CLUSTERS, cluster, PATH_TESTS, testName, PATH_RUNS, testRunName))
        {
            try
            {
                configService.setProperties(
                        null, false, false, runStateStop,
                        PATH_CLUSTERS, cluster, PATH_TESTS, testName, PATH_RUNS, testRunName);
            }
            catch (ZkException e)
            {
                // Ignore
            }
        }
        if (!configService.exists(PATH_CLUSTERS, cluster, PATH_TESTS, testName, PATH_RUNS, testRunName, PATH_CONFIG))
        {
            try
            {
                configService.setString(
                        null, false, false, null,
                        PATH_CLUSTERS, cluster, PATH_TESTS, testName, PATH_RUNS, testRunName, PATH_CONFIG);
            }
            catch (ZkException e)
            {
                // Ignore
            }
        }
        if (!configService.exists(PATH_CLUSTERS, cluster, PATH_TESTS, testName, PATH_RUNS, testRunName, PATH_CONFIG, PATH_RUN_PROPERTIES))
        {
            Properties overrideProperties = emptyProperties;
            try
            {
                configService.setProperties(
                        null, false, false, overrideProperties,
                        PATH_CLUSTERS, cluster, PATH_TESTS, testName, PATH_RUNS, testRunName, PATH_CONFIG, PATH_RUN_PROPERTIES);
            }
            catch (ZkException e)
            {
                // Ignore
            }
        }
    }

    /**
     * Do the actual start work but don't handle any exceptions
     */
    private synchronized void reload()
    {
        // First shut down any previous application context
        if (testRunCtx != null)
        {
            if (testRunCtx.isActive())
            {
                testRunCtx.stop();
                testRunCtx.close();
            }
            testRunCtx = null;
        }
        // Get the parent application context
        AbstractApplicationContext serverCtx = server.getApplicationContext();
        // Only start this component if the parent context is active
        if (!serverCtx.isActive())
        {
            return;
        }
        
        // Start the primary application context, using the server ctx as parent
        testRunCtx = new ClassPathXmlApplicationContext(
                new String[] {"server-zk-context.xml"},
                false);
        // Required for access to the thread pool, for instance, but also ensures that shutdown
        // is passed to the child contexts properly
        testRunCtx.setParent(serverCtx);
        testRunCtx.refresh();
        testRunCtx.start();

        // Get the ZooKeeper configuration
        ConfigService configService = (ConfigService) testRunCtx.getBean("zooKeeperConfigService");
        
        // Prepare any missing data
        createTestStructure(configService);
        
        // Load cluster properties
        Properties clusterProperties = configService.getProperties(
                this,
                PATH_CLUSTERS, cluster, PATH_CLUSTER_PROPERTIES);
        // Load the test properties
        // Note: We have the 'test.properties' below the 'config' so that we can support
        //       multiple properties files one day
        Properties testProperties = configService.getProperties(
                this,
                PATH_CLUSTERS, cluster, PATH_TESTS, testName, PATH_CONFIG, PATH_TEST_PROPERTIES);
        // Load the test definition
        // Note: We have the xxx-context.xml file below 'config' so that we can support
        //       multiple xml context files one day
        String testDef = configService.getString(
                this,
                PATH_CLUSTERS, cluster, PATH_TESTS, testName, PATH_CONFIG, PATH_TEST_CONTEXT);
        // Load test classes
        ClassLoader testClassLoader = configService.getClassLoader(
                this,
                this.getClass().getClassLoader(),
                PATH_CLUSTERS, cluster, PATH_TESTS, testName, PATH_CLASSES);
        // Load test jars
        testClassLoader = configService.getJarLoader(
                this,
                testClassLoader,
                PATH_CLUSTERS, cluster, PATH_TESTS, testName, PATH_JARS);
        // Load test run properties
        Properties testRunProperties = configService.getProperties(
                this,
                PATH_CLUSTERS, cluster, PATH_TESTS, testName, PATH_RUNS, testRunName, PATH_CONFIG, PATH_RUN_PROPERTIES);
        // Load test run controls
        Properties testRunControlProperties = configService.getProperties(
                this,
                PATH_CLUSTERS, cluster, PATH_TESTS, testName, PATH_RUNS, testRunName);
        
        // Combine properties.  Precedence (high to low) ... test run ... test ... cluster
        Properties ctxProperties = new Properties();
        ctxProperties.putAll(clusterProperties);
        ctxProperties.putAll(testProperties);
        ctxProperties.putAll(testRunProperties);
        // Add in guaranteed properties
        ctxProperties.put(PROP_CLUSTER, cluster);
        ctxProperties.put(PROP_SERVER_ID, server.getServerId());
        ctxProperties.put(PROP_TEST_NAME, testName);
        ctxProperties.put(PROP_TEST_RUN_NAME, testRunName);
        ctxProperties.put(PROP_TEST_RUN_FQN, testRunFqn);
        
        // Get the test run control run state
        String controlRunStateStr = testRunControlProperties.getProperty(PROP_CONTROL_RUN_STATE, DEFAULT_RUNSTATE);
        RunState runState = null;
        try
        {
            runState = RunState.valueOf(controlRunStateStr);
        }
        catch (IllegalArgumentException e)
        {
            logger.warn(
                    "\tSet the test run state in '" + testRunFqn + "' to one of " +
                    RunState.PAUSE + " (default), " +
                    RunState.RUN + " or " +
                    RunState.STOP + "; for example, \"{runState:'RUN'}\".");
            runState = RunState.PAUSE;
        }
        logger.info("\tTest run '" + this + "' is in state '" + runState + "'.");
        
        // No child context needed
        if (runState == RunState.STOP)
        {
            return;
        }
        
        // Push test context to a temporary file
        if (testDef == null || testDef.length() == 0)
        {
            logger.error("\tNo application context defined for test: " + testName);
            return;
        }
        File ctxFile = null;
        FileWriter writer = null;
        try
        {
            ctxFile = File.createTempFile(testRunFqn, ".xml");
            writer = new FileWriter(ctxFile);
            writer.append(testDef);
        }
        catch (Throwable e)
        {
            logger.error(
                    "\tFailed to copy test context: \n" +
                    "   Test run: " + testRunFqn,
                    e);
        }
        finally
        {
            try { writer.close(); } catch (Throwable e) {}
        }
        
        // Spin up the context
        FileSystemXmlApplicationContext childCtx = null;
        try
        {
            // Construct the application context, being sure not to auto-start
            childCtx = new FileSystemXmlApplicationContext(
                    new String[] {FILE_PREFIX + ctxFile.getAbsolutePath()},
                    false);
            // Use the server's application context as parent (must be done BEFORE configuring environment)
            childCtx.setParent(testRunCtx);
            // Push cluster properties into the context (must be done AFTER setting parent context)
            ConfigurableEnvironment ctxEnv = childCtx.getEnvironment();
            ctxEnv.getPropertySources().addFirst(
                    new PropertiesPropertySource(
                            ctxProperties.getProperty(PROP_TEST_RUN_FQN),
                            ctxProperties));
            // Make sure that we use the correct ClassLoader
            childCtx.setClassLoader(testClassLoader);
            // Now refresh (to load beans) and start
            childCtx.refresh();
            // We only start it if the mode is RUN
            if (runState == RunState.RUN)
            {
                childCtx.start();
                testRunCtx = childCtx;
            }
        }
        catch (Throwable e)
        {
            logger.error(
                    "\tFailed to start test run context: \n" +
                    "   Test run: " + testRunFqn,
                    e);
            return;
        }
        
        // Remove the context file
        try
        {
            ctxFile.delete();
        }
        catch (Throwable e)
        {
            logger.error("\tFailed to delete temp context file: " + ctxFile, e);
        }
    }
}
