/*
 * Copyright (C) 2005-2012 Alfresco Software Limited.
 *
 * This file is part of Alfresco
 *
 * Alfresco is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Alfresco is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with Alfresco. If not, see <http://www.gnu.org/licenses/>.
 */
package org.alfresco.bm.server;

import java.io.IOException;
import java.io.InputStream;
import java.net.InetAddress;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.StringTokenizer;
import java.util.TreeMap;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import org.I0Itec.zkclient.IZkStateListener;
import org.I0Itec.zkclient.ZkClient;
import org.I0Itec.zkclient.exception.ZkException;
import org.alfresco.config.ConfigChildListener;
import org.alfresco.config.ConfigDataListener;
import org.alfresco.config.ConfigException;
import org.alfresco.config.ConfigService;
import org.alfresco.http.SharedHttpClientProvider;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.apache.commons.cli.PosixParser;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.zookeeper.Watcher.Event.KeeperState;
import org.springframework.beans.BeansException;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.ApplicationListener;
import org.springframework.context.event.ApplicationContextEvent;
import org.springframework.context.event.ContextStartedEvent;
import org.springframework.context.event.ContextStoppedEvent;
import org.springframework.context.support.AbstractApplicationContext;
import org.springframework.context.support.ClassPathXmlApplicationContext;
import org.springframework.core.env.ConfigurableEnvironment;
import org.springframework.core.env.PropertiesPropertySource;

/**
 * Top level Benchmark Server driver class. 
 * 
 * @author Derek Hulley
 * @since 1.0
 */
public class BMServer implements
        ConfigConstants,
        ConfigDataListener,
        ConfigChildListener,
        IZkStateListener,
        ApplicationListener<ApplicationContextEvent>,
        ApplicationContextAware
{
    private static final Log logger = LogFactory.getLog(BMServer.class);
    
    private static final String OPT_RETRY_WAIT = "retryWait";
    private static final String OPT_ATTEMPTS = "attempts";
    private static final String OPT_ZOOKEEPER_URI = "zkUri";
    private static final String OPT_ZOOKEEPER_PATH = "zkPath";
    private static final String OPT_CLUSTER = "cluster";
    private static final String OPT_HELP = "help";
    
    private static final String RESOURCE_SERVER_PROPERTIES = "/server.properties";

    /**
     * Dump usage help
     * 
     * @param options           the command line options
     */
    private static void printUsage(Options options)
    {
        HelpFormatter helpFormatter = new HelpFormatter();
        helpFormatter.printHelp(
                "BMServer [-retryWait <seconds>] [-attempts <number>] " +
                "-zkUri <ZooKeeper URI> -zkPath <ZooKeeper data path> " +
                "-cluster <Server Cluster Name>", options);
    }
    
    /**
     * Real main method
     * 
     * @param args              command line arguments
     */
    public static void main(String ... args)
    {
        logger.info("Server startup parameters: " + Arrays.asList(args));
        
        Options options = new Options();
        options.addOption(new Option(OPT_RETRY_WAIT, true, "Number of seconds to wait between startup attempts"));
        options.addOption(new Option(OPT_ATTEMPTS, true, "Number of times to try to start before quitting"));
        options.addOption(new Option(OPT_ZOOKEEPER_URI, true, "The ZooKeeper server list e.g. '127.0.0.1:3000,127.0.0.1:3001,127.0.0.1:3002'"));
        options.addOption(new Option(OPT_ZOOKEEPER_PATH, true, "The ZooKeeper data path e.g. '/alfresco/bm'"));
        options.addOption(new Option(OPT_CLUSTER, true, "Specify a non-default cluster name to join.  The server version is normally used e.g. '1.3_SNAPHOST'." +
                                                        "  The server version should be included in the cluster name, in any case."));
        options.addOption(new Option("h", "help", false, "This help message"));
        
        long retryWaitS = 0L;
        int attempts = 0;
        String zkUri = null;
        String zkPath = null;
        String cluster = null;
        InputStream serverPropertiesIs = null;
        try
        {
            CommandLineParser cmdLineParser = new PosixParser();
            CommandLine cmdLine = cmdLineParser.parse(options, args, true);
            if (cmdLine.hasOption(OPT_HELP))
            {
                printUsage(options);
                return;
            }
            retryWaitS = Long.parseLong(cmdLine.getOptionValue(OPT_RETRY_WAIT, "30"));
            if (retryWaitS <= 0)
            {
                retryWaitS = 1;
            }
            attempts = Integer.parseInt(cmdLine.getOptionValue(OPT_ATTEMPTS, "1"));
            zkUri = cmdLine.getOptionValue(OPT_ZOOKEEPER_URI, "127.0.0.1:2181");
            if (System.getProperty(OPT_ZOOKEEPER_URI) != null)
            {
                zkUri = System.getProperty(OPT_ZOOKEEPER_URI);
            }
            zkPath = cmdLine.getOptionValue(OPT_ZOOKEEPER_PATH, "/alfresco/bm");
            if (System.getProperty(OPT_ZOOKEEPER_PATH) != null)
            {
                zkPath = System.getProperty(OPT_ZOOKEEPER_PATH);
            }
            // Look up the default cluster name
            Properties serverProperties = new Properties();
            serverPropertiesIs = BMServer.class.getResourceAsStream(RESOURCE_SERVER_PROPERTIES);
            if (serverPropertiesIs == null)
            {
                throw new IOException("Server version not found.");
            }
            serverProperties.load(serverPropertiesIs);
            String version = serverProperties.getProperty("server.version");
            cluster = cmdLine.getOptionValue(OPT_CLUSTER, version);
            if (System.getProperty(OPT_CLUSTER) != null)
            {
                cluster = System.getProperty(OPT_CLUSTER);
            }
            cluster = cluster.replaceAll("-", "_");                       // Cluster name may not contain '-' (MongoDB)
        }
        catch (NumberFormatException e)
        {
            printUsage(options);
            return;
        }
        catch (IOException e)
        {
            System.err.println("Unable to read 'version' from resource: " + RESOURCE_SERVER_PROPERTIES);
            return;
        }
        catch (ParseException e)
        {
            printUsage(options);
            return;
        }
        finally
        {
            if (serverPropertiesIs != null)
            {
                try { serverPropertiesIs.close(); } catch (Throwable e) {}
            }
        }
        
        int counter = attempts;
        while (counter > 0)
        {
            logger.info("Starting server.");
            try
            {
                RunState instruction = run(zkUri, zkPath, cluster);
                // Having exited normally, what should we do next?
                switch (instruction)
                {
                    case STOP:
                        // We're done
                        logger.info("Server stopping (explicit STOP received).");
                        return;
                    case RESTART:
                    case RUN:
                        // Go around again; reset the counter
                        counter = attempts;
                        break;
                    default:
                        throw new IllegalStateException("Unexpected instruction: " + instruction);
                }
            }
            catch (Throwable e)
            {
                logger.error("Failed to start server context.", e);
                // Have another go, depending on the counter
                counter--;
                if (counter > 0)
                {
                    long retryWaitMs = retryWaitS * 1000L;
                    logger.info("Server will attempt a restart in " + retryWaitS + "s.  " + counter + " attempts remaining.");
                    synchronized(zkUri)
                    {
                        // Wait unless we are not going to try again
                        try { zkUri.wait(retryWaitMs); } catch (InterruptedException ee) {}
                    }
                }
            }
        }
    }
    
    /**
     * Main method that can exit and trigger a full, ground-up rerun
     * 
     * @return              an instruction on what to do next
     */
    private static RunState run(String zkUri, String zkPath, String cluster) throws Throwable
    {
        // Push initial configuration to the system properties
        System.setProperty(OPT_ZOOKEEPER_URI, zkUri);
        System.setProperty(OPT_ZOOKEEPER_PATH, zkPath);
        System.setProperty(OPT_CLUSTER, cluster);
        
        // Get cluster startup configuration
        Properties clusterConfig = getClusterStartupProperties(cluster);
        
        // Construct the application context, being sure not to auto-start
        ClassPathXmlApplicationContext ctx = new ClassPathXmlApplicationContext(
                new String[] {"server-context.xml"},
                false);
        // Push cluster properties into the context
        ConfigurableEnvironment ctxEnv = ctx.getEnvironment();
        ctxEnv.getPropertySources().addFirst(new PropertiesPropertySource("BMServerProperties", clusterConfig));
        ctx.registerShutdownHook();
        // Now refresh (to load beans) and start
        ctx.refresh();
        ctx.start();
        
        // The server has been started
        logger.info("Server started.");
        
        try
        {
            // Get the server bean
            BMServer server = (BMServer) ctx.getBean("server");
            while (true)
            {
                // We wait on it, either for notifications or just for regular checks
                synchronized (server)
                {
                    try { server.wait(5000L); } catch (InterruptedException e) {}
                }
                RunState desiredRunState = server.getDesiredRunState();
                RunState actualRunState = server.getRunState();
                if (desiredRunState == RunState.STOP || desiredRunState != actualRunState)
                {
                    // The run state needs to change
                    return desiredRunState;
                }
                // The server is still running as before.  Prompt it to check the test runs.
                server.checkTestRunsForRestart();
            }
        }
        finally
        {
            try
            {
                ctx.stop();
                ctx.close();
            }
            catch (Throwable e)
            {
                logger.error("Error closing application context.", e);
            }
        }
    }
    
    /**
     * Creates necessary nodes, avoiding any concurrency conditions when running
     * multiple servers in the same cluster configuration.
     */
    private static void createConfigStructure(ConfigService configService, String cluster)
    {
        Properties defaultProperties = new Properties();
        defaultProperties.setProperty(PROP_MONGO_URI,                   "mongodb://127.0.0.1:27017/alfrescobm");
        defaultProperties.setProperty(PROP_MONGO_AUTOCONNECTRETRY,      "true");
        defaultProperties.setProperty(PROP_MONGO_CONNECTIONSPERHOST,    "${eventProcessorThreads}");
        defaultProperties.setProperty(PROP_MONGO_SOCKETTIMEOUT,         "600000");
        defaultProperties.setProperty(PROP_MONGO_WRITENUMBER,           "1");
        defaultProperties.setProperty(PROP_EVENT_PROCESSOR_THREADS,     "16");
        defaultProperties.setProperty(PROP_HTTP_CONNECTION_MAX,         "${eventProcessorThreads}");
        defaultProperties.setProperty(PROP_HTTP_CONNECTION_TIMEOUT_MS,  "" + SharedHttpClientProvider.DEFAULT_CONNECTION_TIMEOUT_MILLISEC);
        defaultProperties.setProperty(PROP_HTTP_SOCKET_TIMEOUT_MS,      "" + SharedHttpClientProvider.DEFAULT_SOCKET_TIMEOUT_MILLISEC);
        defaultProperties.setProperty(PROP_HTTP_SOCKET_TTL_MS,          "" + SharedHttpClientProvider.DEFAULT_SOCKET_TTL_MILLISEC);
        
        Properties runStatePause = new Properties();
        runStatePause.setProperty(PROP_CONTROL_RUN_STATE, RunState.RUN.toString());
        
        if (!configService.exists(PATH_CLUSTERS))
        {
            try
            {
                configService.setString(
                        null, false, false, null,
                        PATH_CLUSTERS);
            }
            catch (ZkException e)
            {
                // Ignore
            }
        }
        if (!configService.exists(PATH_CLUSTERS, cluster))
        {
            try
            {
                configService.setProperties(
                        null, false, false, runStatePause,
                        PATH_CLUSTERS, cluster);
            }
            catch (ZkException e)
            {
                // Ignore
            }
        }
        if (!configService.exists(PATH_CLUSTERS, cluster, PATH_CLUSTER_PROPERTIES))
        {
            try
            {
                configService.setProperties(
                        null, false, false, defaultProperties,
                        PATH_CLUSTERS, cluster, PATH_CLUSTER_PROPERTIES);
            }
            catch (ZkException e)
            {
                // Ignore
            }
        }
        if (!configService.exists(PATH_CLUSTERS, cluster, PATH_LOADED))
        {
            try
            {
                configService.setString(
                        null, false, false, null,
                        PATH_CLUSTERS, cluster, PATH_LOADED);
            }
            catch (ZkException e)
            {
                // Ignore
            }
        }
        if (!configService.exists(PATH_CLUSTERS, cluster, PATH_SERVERS))
        {
            try
            {
                configService.setString(
                        null, false, false, null,
                        PATH_CLUSTERS, cluster, PATH_SERVERS);
            }
            catch (ZkException e)
            {
                // Ignore
            }
        }
        if (!configService.exists(PATH_CLUSTERS, cluster, PATH_TESTS))
        {
            try
            {
                configService.setString(
                        null, false, false, null,
                        PATH_CLUSTERS, cluster, PATH_TESTS);
            }
            catch (ZkException e)
            {
                // Ignore
            }
        }
    }

    /**
     * Get the startup properties from ZooKeeper
     * 
     * @param cluster   the ID of the server cluster
     * @return          the startup properties (never <tt>null</tt>)
     * @throws ConfigException if the configuration could not be loaded
     */
    private static Properties getClusterStartupProperties(String cluster)
    {
        ClassPathXmlApplicationContext ctx = null;
        // Get the data from the root path node
        try
        {
            // Start the context
            ctx = new ClassPathXmlApplicationContext("server-zk-context.xml");
            ctx.start();
            // Get the ZooKeeper configuration
            ConfigService configService = (ConfigService) ctx.getBean("zooKeeperConfigService");
            // Ensure configuration is available
            createConfigStructure(configService, cluster);            
            // Get properties
            Properties config = configService.getProperties(
                    null,
                    PATH_CLUSTERS, cluster, PATH_CLUSTER_PROPERTIES);
            // Done
            if (logger.isDebugEnabled())
            {
                logger.debug(
                        "Loaded cluster configuration: \n" +
                        "   cluster: " + cluster + "\n" +
                        "   Config:  " + config);
            }
            return config;
        }
        finally
        {
            if (ctx != null)
            {
                ctx.stop();
                ctx.close();
            }
        }
    }
    
    //-- Instance implementation --//
    
    private final ConfigService configService;
    /** The cluster that this server will join */
    private final String cluster;
    /** A unique ID for the server instance */
    private String serverId;
    /** the server's application context as populated by Spring */
    private AbstractApplicationContext ctx;
    /** The desired run state of the server. */
    private RunState desiredRunState;
    /** The actual run state of the server. */
    private RunState runState;
    
    /** Keep track of all test runs */
    private final Map<BMTestRun.Key, BMTestRun> testRuns;
    
    /**
     * Constructor as used by Spring context
     */
    public BMServer(ConfigService configService, String cluster)
    {
        this.configService = configService;
        this.cluster = cluster;
        this.serverId = DEFAULT_SERVER_ID;
        
        // Check and set run states
        if (runState == RunState.RESTART)
        {
            throw new ConfigException("Valid 'runState' values are 'RUN', 'PAUSE' and 'STOP'.");
        }
        this.runState = RunState.PAUSE;
        this.desiredRunState = runState;
        
        // Data records of specific run
        this.testRuns = new TreeMap<BMTestRun.Key, BMTestRun>();
    }

    /**
     * Get the name of the cluster that the server will join
     * 
     * @return              Returns one of <b>{zkPath}/clusters/*</b>
     */
    public String getCluster()
    {
        return cluster;
    }

    /**
     * @return              the run state for the server as set at startup
     */
    public RunState getRunState()
    {
        return runState;
    }

    /**
     * @return              get the desired run state for the server
     */
    public synchronized RunState getDesiredRunState()
    {
        return desiredRunState;
    }

    /**
     * Sets the desired run state and notifies listeners (main thread).
     * 
     * @param runState      the desired server run state
     */
    public synchronized void setDesiredRunState(RunState runState)
    {
        this.desiredRunState = runState;
        this.notifyAll();
    }

    /**
     * Get the application context that the server is using
     * 
     * @return              the server's application context or <tt>null</tt> if it has not been set
     */
    public AbstractApplicationContext getApplicationContext()
    {
        return ctx;
    }

    /**
     * Stores the application context for use by the tests.
     * 
     * @param applicationContext    the server's application context
     */
    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException
    {
        this.ctx = (AbstractApplicationContext) applicationContext;
    }

    /**
     * @return              Returns the unique server identifier
     */
    public String getServerId()
    {
        return serverId;
    }

    /**
     * Triggers a {@link RunState#RESTART restart}
     */
    @Override
    public synchronized void handleStateChanged(KeeperState state) throws Exception
    {
        logger.error("ZooKeeper state has changed (" + state + "); triggering restart.");
        setDesiredRunState(RunState.RESTART);
    }

    @Override
    public synchronized void handleNewSession() throws Exception
    {
        logger.debug("New session.");
    }
    
    /**
     * Prompt the server to perform a check of the loaded test runs in case they have changed state.
     * <p/>
     * It is necessary to do this from an external source because ZooKeeper-prompted actions
     * cannot shut down the ZooKeeper instance.
     */
    public synchronized void checkTestRunsForRestart()
    {
        for (Map.Entry<BMTestRun.Key, BMTestRun> entry : testRuns.entrySet())
        {
            BMTestRun testRun = entry.getValue();
            testRun.checkForRestart();
        }
    }

    /**
     * Triggers a {@link #setDesiredRunState(org.alfresco.bm.config.ConfigConstants.RunState) restart}
     * whenever the cluster configuration change is made.  The entire server configuration is invalid.
     */
    @Override
    public synchronized void dataChanged(String path, boolean deleted)
    {
        if (path.endsWith(PATH_CLUSTERS + SEPARATOR + cluster + SEPARATOR + PATH_CLUSTER_PROPERTIES))
        {
            // Basic properties changed: Restart.
            setDesiredRunState(RunState.RESTART);
        }
        else if (path.endsWith(PATH_CLUSTERS + SEPARATOR + cluster))
        {
            // RunState changed: Restart.
            setDesiredRunState(RunState.RESTART);
        }
        else if (path.endsWith(PATH_CLUSTERS + SEPARATOR + cluster + SEPARATOR + PATH_LOADED))
        {
            // List of tests to run has changed.  Reload tests.
            reloadTestRuns();
        }
        else if (path.endsWith(PATH_CLUSTERS + SEPARATOR + cluster + SEPARATOR + PATH_SERVERS + SEPARATOR + serverId))
        {
            if (deleted)
            {
                // Server ID was removed: SHUTDOWN
                setDesiredRunState(RunState.STOP);
            }
            else
            {
                // Server ID was changed: IGNORE
            }
        }
    }

    @Override
    public synchronized void childrenChanged(String path)
    {
        if (path.contains(SEPARATOR + PATH_RUNS))
        {
            // New test run added?
            reloadTestRuns();
        }
        else if (path.contains(SEPARATOR + PATH_TESTS))
        {
            // New test added?
            reloadTestRuns();
        }
        else
        {
            throw new UnsupportedOperationException("Received children change for unexpected path: " + path);
        }
    }

    /**
     * Starts up application contexts as required.
     */
    @Override
    public synchronized void onApplicationEvent(ApplicationContextEvent event)
    {
        // Ignore events from child application contexts
        if (event.getSource() != this.ctx)
        {
            // The event came from an child context
            return;
        }
        
        if (event instanceof ContextStartedEvent)
        {
            // Register for any loss of the ZooKeeper session
            ZkClient zk = (ZkClient) ctx.getBean("zooKeeper");
            zk.subscribeStateChanges(this);
            
            // Register an interest in any property changes
            configService.getProperties(
                    this,
                    PATH_CLUSTERS, cluster, PATH_CLUSTER_PROPERTIES);
            
            // Get the control properties
            Properties controlProperties = configService.getProperties(
                    this,
                    PATH_CLUSTERS, cluster);
            String runStateStr = controlProperties.getProperty(PROP_CONTROL_RUN_STATE);
            try
            {
                runState = RunState.valueOf(runStateStr);
            }
            catch (IllegalArgumentException e)
            {
                logger.warn(
                        "Set the cluster run state in '<zkPath>/clusters/" + cluster + " to one of " +
                        RunState.PAUSE + " (default), " +
                        RunState.RUN + " or " +
                        RunState.STOP + "; for example, \"{runState:'RUN'}\".");
                this.runState = RunState.PAUSE;
            }
            desiredRunState = runState;
            
            // Create an ephemeral node for the server
            String ipAddress = "unknown";
            String hostName = "unknown";
            try
            {
                ipAddress = InetAddress.getLocalHost().toString();
                hostName = InetAddress.getLocalHost().getHostName();
            }
            catch (Throwable e)
            {
                // Just ignore
            }
            Properties serverProps = new Properties();
            serverProps.put("ipAddress", ipAddress);
            serverProps.put("hostName", hostName);
            serverId = configService.setProperties(
                    this,
                    true, true, serverProps,
                    PATH_CLUSTERS + SEPARATOR + cluster + SEPARATOR + PATH_SERVERS,
                    (hostName + "_"));             // e.g. {zkRoot}/clusters/{cluster}/servers/JUPITER5_00000001
            
            // Start the work-horse contexts
            reloadTestRuns();
        }
        else if (event instanceof ContextStoppedEvent)
        {
            stopTestRuns();
            // Notify the thread pool executor that it should stop
            try
            {
                ThreadPoolExecutor executor = (ThreadPoolExecutor) ctx.getBean("eventExecutor");
                executor.shutdownNow();
                executor.awaitTermination(15, TimeUnit.SECONDS);
            }
            catch (InterruptedException e)
            {
            }
        }
    }
    
    /**
     * Find and start, if necessary, all test run contexts.  Any existing contexts that
     * are not present using the new configuration will be stopped.
     * <p/>
     * Note that nothing is done unless the {@link #getRunState() run state} is {@link RunState#RUN}.
     */
    private synchronized void reloadTestRuns()
    {
        if (getRunState() != RunState.RUN)
        {
            // We stop them all
            stopTestRuns();
            return;
        }
        
        // Get the test runs to load
        String loadedTestRunsStr = configService.getString(this, PATH_CLUSTERS, cluster, PATH_LOADED);
        loadedTestRunsStr = (loadedTestRunsStr == null) ? "" : loadedTestRunsStr;
        // Parse it
        StringTokenizer st = new StringTokenizer(loadedTestRunsStr, ",");
        Set<BMTestRun.Key> proposedTestRunKeys = new HashSet<BMTestRun.Key>(17);
        while (st.hasMoreTokens())
        {
            String loadedTestRunStr = st.nextToken().trim();
            String[] loadedTestRunParts = loadedTestRunStr.split("\\.");
            if (loadedTestRunParts.length != 2 || loadedTestRunParts[0].length() < 1 || loadedTestRunParts[1].length() < 1)
            {
                logger.warn("Ignoring loaded test run, '" + loadedTestRunStr + "'.  The format is 'test1.run1'.");
                continue;
            }
            proposedTestRunKeys.add(new BMTestRun.Key(loadedTestRunParts[0], loadedTestRunParts[1]));
        }
        
        // Now find all test runs that need to be stopped
        Set<BMTestRun.Key> toStopTestRunKeys = new HashSet<BMTestRun.Key>(testRuns.keySet());  // Copy against mod
        toStopTestRunKeys.removeAll(proposedTestRunKeys);
        // Shut them down
        for (BMTestRun.Key toStopTestRunKey : toStopTestRunKeys)
        {
            BMTestRun toStopTestRun = testRuns.get(toStopTestRunKey);
            toStopTestRun.stop();
            testRuns.remove(toStopTestRunKey);
        }
        
        // Remove any runs that need to remain
        Set<BMTestRun.Key> toStartTestRunKeys = new HashSet<BMTestRun.Key>(proposedTestRunKeys);  // Copy against mod
        toStartTestRunKeys.removeAll(testRuns.keySet());
        
        // Now reload all the runs that are present
        reloadTestRuns(toStartTestRunKeys);
    }
    
    private synchronized void reloadTestRuns(Set<BMTestRun.Key> testRunKeys)
    {
        for (BMTestRun.Key testRunKey : testRunKeys)
        {
            // Construct the test run
            BMTestRun testRun = new BMTestRun(this, testRunKey);
            // Start it
            testRuns.put(testRunKey, testRun);
            testRun.start();
        }
    }
    
    /**
     * Stop all test runs.  Note that this does not mean that they were active.
     */
    private synchronized void stopTestRuns()
    {
        // Stop them all
        for (BMTestRun testRun : testRuns.values())
        {
            testRun.stop();
        }
    }
}
