package org.alfresco.bm.event.selector;

import java.util.List;
import java.util.NavigableMap;
import java.util.Random;
import java.util.StringTokenizer;
import java.util.TreeMap;

import org.alfresco.bm.event.EventProcessorRegistry;

/**
 * Select a successor event from a set of relatively weighted event successors.
 * 
 * Based on http://stackoverflow.com/a/6409791.
 * 
 * @author steveglover
 *
 */
public class RandomWeightedEventSelector extends AbstractEventSelector
{
    private final NavigableMap<Double, EventSuccessor> eventSuccessors = new TreeMap<Double, EventSuccessor>();

    private final Random random;
    private double total = 0;

    public RandomWeightedEventSelector(EventProcessorRegistry registry, List<String> eventSuccessors)
    {
        super(registry);
        this.random = new Random(System.currentTimeMillis());

        for(String eventSuccessorInfo : eventSuccessors)
        {
            StringTokenizer st = new StringTokenizer(eventSuccessorInfo, ",");
            if(st.countTokens() < 2)
            {
                throw new RuntimeException("Event successor must have an event name and relative weighting separated by a comma");
            }
            String eventName = st.nextToken();
            int weighting = Integer.parseInt(st.nextToken());
            EventSuccessor eventSuccessor = new EventSuccessor(eventName, weighting);
            add(eventSuccessor.getWeighting(), eventSuccessor);
        }
    }

    /**
     * Add an event successor to the list.
     * 
     * @param weight
     * @param result
     */
    private void add(double weight, EventSuccessor result)
    {
        if (weight <= 0) return;
        total += weight;
        eventSuccessors.put(total, result);
    }

    @Override
    protected EventSuccessor next()
    {
        double value = random.nextDouble() * total;
        return eventSuccessors.ceilingEntry(value).getValue();
    }

    @Override
    public int size()
    {
        return eventSuccessors != null ? eventSuccessors.size() : 0;
    }
}
