package org.alfresco.bm.event.selector;

import org.alfresco.bm.event.Event;
import org.alfresco.bm.event.EventProcessor;
import org.alfresco.bm.event.EventProcessorRegistry;

/**
 * Abstract base class for event selectors. 
 *  
 * @author steveglover
 *
 */
public abstract class AbstractEventSelector implements EventSelector
{
    private EventProcessorRegistry registry;

    public AbstractEventSelector(EventProcessorRegistry registry)
    {
        super();
        this.registry = registry;
    }
    
    /**
     * Implemented by subclasses to select a successor event.
     * 
     * @return the successor event
     */
    protected abstract EventSuccessor next();

    @Override
    public Event nextEvent(Object input, Object response) throws Exception
    {
        Event ret = null;

        if(size() > 0)
        {
            EventSuccessor eventSuccessor = next();
            String nextEventName = eventSuccessor.getEventName();

            if (nextEventName != null && !nextEventName.equals("") && !nextEventName.equalsIgnoreCase("noop"))
            {
                EventProcessor eventProcessor = (EventProcessor)registry.getProcessor(nextEventName);
                if(eventProcessor == null)
                {
                    throw new RuntimeException(
                            "Event selector contains unknown event mapping: " + nextEventName + "." +
                            "No further events will be published.");
                }
                else if (!(eventProcessor instanceof EventDataCreator))
                {
                    throw new RuntimeException("Unable to cast event processor " + nextEventName + " to CreateData");
                }
                else
                {
                    EventDataCreator eventDataCreator = (EventDataCreator)eventProcessor;
                    Object dataObject = eventDataCreator.createDataObject(input, response);
                    if (dataObject != null)
                    {
                        ret = new Event(nextEventName, 0L, dataObject);
                    }
                }
            }
            else
            {
                // nothing to do, noop event
            }
        }
        else
        {
            // nothing to do, no successor events
        }

        return ret;
    }    
}
