/*
 * Copyright (C) 2005-2012 Alfresco Software Limited.
 *
 * This file is part of Alfresco
 *
 * Alfresco is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Alfresco is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with Alfresco. If not, see <http://www.gnu.org/licenses/>.
 */
package org.alfresco.bm.event.mongo;

import java.util.List;

import org.alfresco.bm.event.AbstractResultService;
import org.alfresco.bm.event.EventRecord;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.data.domain.Sort;
import org.springframework.data.domain.Sort.Direction;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;

/**
 * A Mongo-based implementation of the results for benchmark test runs.
 * 
 * @author Derek Hulley
 * @since 1.0
 */
public class MongoResultService extends AbstractResultService
{
    private static Log logger = LogFactory.getLog(MongoResultService.class);

    private final MongoTemplate mongo;
    private final String collection;

    /**
     * Construct a provider that works using the given Mongo instance and
     * collection name.
     * 
     * @param mongo the MongoDB template to use
     * @param collection the name of the MongoDB collection
     */
    public MongoResultService(MongoTemplate mongo, String collection)
    {
        this.mongo = mongo;
        this.collection = collection;
        // Initialize indexes
        EventRecord.checkIndexes(mongo, collection);
    }
    
    @Override
    public String getDataLocation()
    {
        return mongo.getDb().getMongo() + "/" + mongo.getDb().getName() + "/" + collection;
    }

    @Override
    public EventRecord recordResult(EventRecord result)
    {
        mongo.insert(result, collection);
        // Done
        if (logger.isDebugEnabled())
        {
            logger.debug("Recorded result: " + result);
        }
        return result;
    }

    @Override
    public List<EventRecord> getResults(String eventName, int skip, int limit)
    {
        Criteria criteria = Criteria
                .where(EventRecord.FIELD_EVENT_NAME).is(eventName);
        Query query = new Query(criteria).skip(skip).limit(limit);
        query.with(new Sort(Direction.ASC, EventRecord.FIELD_START_TIME));
        List<EventRecord> results = mongo.find(query, EventRecord.class, collection);
        // Done
        if (logger.isDebugEnabled())
        {
            logger.debug("\n" +
                    "Found results: \n" +
                    "   Event name:     " + eventName + "\n" +
                    "   Skip:           " + skip + "\n" +
                    "   Limit:          " + limit + "\n" +
                    "   Results:        " + results);
        }
        return results;
    }

    @Override
    public List<EventRecord> getResults(
            long startTime,
            String eventName,
            Boolean successOrFail,
            int skip, int limit)
    {
        Criteria criteria = Criteria
                .where(EventRecord.FIELD_START_TIME).gte(startTime);
        if (eventName != null)
        {
            criteria.and(EventRecord.FIELD_EVENT_NAME).is(eventName);
        }
        if (successOrFail != null)
        {
            criteria.and(EventRecord.FIELD_SUCCESS).is(successOrFail);
        }
        Query query = new Query(criteria).skip(skip).limit(limit);
        query.with(new Sort(Direction.ASC, EventRecord.FIELD_START_TIME));
        List<EventRecord> results = mongo.find(query, EventRecord.class, collection);
        // Done
        if (logger.isDebugEnabled())
        {
            logger.debug("\n" +
                    "Found results: \n" +
                    "   Start:          " + startTime + "\n" +
                    "   Event name:     " + eventName + "\n" +
                    "   Success/fail:   " + successOrFail + "\n" +
                    "   Skip:           " + skip + "\n" +
                    "   Limit:          " + limit + "\n" +
                    "   Results:        " + results);
        }
        return results;
    }

    @Override
    public List<String> getEventNames()
    {
        @SuppressWarnings("unchecked")
        List<String> results = mongo.getCollection(collection).distinct(EventRecord.FIELD_EVENT_NAME);
        return results;
    }

    @Override
    public long countEventsByName(String name)
    {
        Criteria criteria = Criteria.where(EventRecord.FIELD_EVENT_NAME).is(name);
        Query query = new Query(criteria);
        
        long count = mongo.count(query, collection);
        if(logger.isDebugEnabled())
        {
            logger.debug("Counted " + count + " results for Event name: " + name);
        }
        return count;
    }

    @Override
    public long countEventsBySuccess()
    {
        Criteria criteria = Criteria.where(EventRecord.FIELD_SUCCESS).is(Boolean.TRUE);
        Query query = new Query(criteria);
        
        long count = mongo.count(query, collection);
        if(logger.isDebugEnabled())
        {
            logger.debug("Counted " + count + " results for success.");
        }
        return count;
    }

    @Override
    public long countEventsByFailure()
    {
        Criteria criteria = Criteria.where(EventRecord.FIELD_SUCCESS).is(Boolean.FALSE);
        Query query = new Query(criteria);
        
        long count = mongo.count(query, collection);
        if(logger.isDebugEnabled())
        {
            logger.debug("Counted " + count + " results for failure.");
        }
        return count;
    }
    
    @Override
    public long getMinStartTime()
    {
        Query query = new Query();
        query.with(new Sort(Direction.ASC, EventRecord.FIELD_START_TIME));
        query.limit(1);
        
        EventRecord firstResult = (EventRecord) mongo.findOne(query, EventRecord.class, collection);
        return firstResult == null ? System.currentTimeMillis() : firstResult.getStartTime();
    }

    @Override
    public long getMaxStartTime()
    {
        Query query = new Query();
        query.with(new Sort(Direction.DESC, EventRecord.FIELD_START_TIME));
        query.limit(1);
        
        EventRecord firstResult = (EventRecord) mongo.findOne(query, EventRecord.class, collection);
        return firstResult == null ? System.currentTimeMillis() : firstResult.getStartTime();
    }
}
