/*
 * Copyright (C) 2005-2012 Alfresco Software Limited.
 *
 * This file is part of Alfresco
 *
 * Alfresco is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Alfresco is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with Alfresco. If not, see <http://www.gnu.org/licenses/>.
 */
package org.alfresco.bm.event.mongo;

import java.util.List;

import org.alfresco.bm.event.AbstractEventService;
import org.alfresco.bm.event.Event;
import org.alfresco.bm.event.EventService;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.data.domain.Sort;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;

/**
 * An {@link EventService} <b>MongoDB</b> collection
 * 
 * @author Derek Hulley
 * @since 1.0
 */
public class MongoEventService extends AbstractEventService
{
    private static Log logger = LogFactory.getLog(MongoEventService.class);

    private final MongoTemplate mongo;
    private final String collection;

    /**
     * Construct a provider that works using the given Mongo instance and
     * collection name.
     * 
     * @param mongo the MongoDB template to use
     * @param collection the name of the MongoDB collection
     */
    public MongoEventService(MongoTemplate mongo, String collection)
    {
        this.mongo = mongo;
        this.collection = collection;
        // Initialize indexes
        Event.checkIndexes(mongo, collection);
    }

    @Override
    public long count()
    {
        return mongo.count(null, collection);
    }

    @Override
    public List<Event> findEvents(String name, int skip, int limit)
    {
        Criteria criteria = Criteria
                .where(Event.FIELD_NAME).is(name);
        Query query = new Query(criteria).skip(skip).limit(limit);
        
        // Query
        List<Event> events = mongo.find(query, Event.class, collection);
        // Done
        if (logger.isDebugEnabled())
        {
            logger.debug("\n" +
                    "Found events: \n" +
                    "   Event name: " + name + "\n" +
                    "   Events:     " + events);
        }
        return events;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Event nextEvent(String dataOwner, long latestScheduledTime, long latestQueueTime)
    {
        Criteria criteria = Criteria
                .where(Event.FIELD_LOCK_OWNER).is(null)
                .and(Event.FIELD_SCHEDULED_TIME).lte(new Long(latestScheduledTime))
                .and(Event.FIELD_QUEUE_TIME).lte(new Long(latestQueueTime))
                .orOperator(
                        Criteria.where(Event.FIELD_DATA_OWNER).is(null),
                        Criteria.where(Event.FIELD_DATA_OWNER).is(dataOwner));
        Query query = new Query(criteria);
        query.with(new Sort(Event.FIELD_SCHEDULED_TIME));

        long now = System.currentTimeMillis();
        Update update = new Update()
                .set(Event.FIELD_QUEUE_TIME, now);

        Event event = mongo.findAndModify(query, update, Event.class, collection);
        // Make sure we return the event, as modified
        if (event != null)
        {
            event.setQueueTime(now);
        }
        // Done
        if (logger.isDebugEnabled())
        {
            logger.debug("\n" +
                    "Fetched next event (no lock present): \n" +
                    "   Latest scheduled time:  " + latestScheduledTime + "\n" +
                    "   Latest queue time:      " + latestQueueTime + "\n" +
                    "   Event: " + event);
        }
        return event;
    }
    
    /**
     * {@inheritDoc}
     */
    @Override
    public Event lockEvent(Event event, String lockOwner)
    {
        Criteria criteria = Criteria
                .where(Event.FIELD_LOCK_OWNER).is(null)
                .and(Event.FIELD_ID).is(event.getId());
        
        Query query = new Query(criteria);
        
        long now = System.currentTimeMillis();
        Update update = new Update()
                .set(Event.FIELD_LOCK_OWNER, lockOwner)
                .set(Event.FIELD_LOCK_TIME, now);
        
        // Query
        Event eventUpdate = mongo.findAndModify(query, update, Event.class, collection);
        if (eventUpdate != null)
        {
            // Get the modified version
            event = mongo.findById(eventUpdate.getId(), Event.class, collection);
            // Done
            if (logger.isDebugEnabled())
            {
                logger.debug("\n" +
                        "Locked event: \n" +
                        "   Owner: " + lockOwner + "\n" +
                        "   Event: " + event);
            }
            return event;
        }
        else
        {
            // Done
            if (logger.isDebugEnabled())
            {
                logger.debug("\n" +
                        "Failed to get event lock: \n" +
                        "   Owner: " + lockOwner + "\n" +
                        "   Event: " + event);
            }
            return null;
        }
    }
    
    @Override
    public Event getEvent(String id)
    {
        Criteria criteria = Criteria.where(Event.FIELD_ID).is(id);
        Query query = new Query(criteria);
        return mongo.findOne(query, Event.class, collection);
    }

    @Override
    public boolean deleteEvent(Event event)
    {
        Criteria criteria = Criteria.where(Event.FIELD_ID).is(event.getId());
        Query query = new Query(criteria);
        Event deletedEvent = mongo.findAndRemove(query, Event.class, collection);
        if (deletedEvent == null)
        {
            return false;
        }
        else
        {
            return true;
        }
    }

    @Override
    public Event putEvent(Event event)
    {
        mongo.insert(event, collection);
        // The event will be updated by the MongoTemplate
        return event;
    }
}
