/*
 * Copyright (C) 2005-2012 Alfresco Software Limited.
 *
 * This file is part of Alfresco
 *
 * Alfresco is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Alfresco is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with Alfresco. If not, see <http://www.gnu.org/licenses/>.
 */
package org.alfresco.bm.event;

import java.io.Serializable;
import java.util.Collections;
import java.util.List;

import org.apache.commons.lang.exception.ExceptionUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * A unit of work that can be executed by the event processing threads.
 * It delegates to the
 * {@link EventService#EventService(String, String, Event, EventProcessor, EventService) event processor},
 * which does not need to know anything about the events or event recording.
 * 
 * @author Derek Hulley
 * @since 1.0
 */
public class EventWork implements Runnable
{
    private static final Log logger = LogFactory.getLog(EventService.class);
    
    private final String serverId;
    private final String testRunFqn;
    private final Event event;
    private final EventProcessor processor;
    private final EventService eventService;
    private final ResultService resultService;
    
    /**
     * Construct work to be executed by a thread
     * 
     * @param serverId      the identifier of the server process handling the event
     * @param testRunFqn    the fully qualified name of the test run initiating the work
     * @param event         the event to be processed
     * @param processor     the component that will do the actual processing
     * @param eventService  the queue events that will be updated with new events
     * @param resultService the service to store results of the execution
     */
    public EventWork(
            String serverId, String testRunFqn,
            Event event,
            EventProcessor processor, EventService eventService, ResultService resultService)
    {
        this.serverId = serverId;
        this.testRunFqn = testRunFqn;
        this.event = event;
        this.processor = processor;
        this.eventService = eventService;
        this.resultService = resultService;
    }

    @Override
    public void run()
    {
        // Set the start and end times for the event
        long before = System.currentTimeMillis();
        long warnDelay = processor.getWarnDelay();
        boolean chart = processor.isChart();

        // Grab the event
        try
        {
            // Update event times
            Event lockedEvent = eventService.lockEvent(event, serverId);
            if (lockedEvent == null)
            {
                if (logger.isDebugEnabled())
                {
                    logger.debug("Event discarded.  Lock taken: " + event);
                }
                return;
            }
        }
        catch (Throwable e)
        {
            logger.error("Failure while attempting to lock: " + event, e);
            return;
        }
        
        EventResult result = null;
        try
        {
            // Process the event
            result = processor.processEvent(event);
        }
        catch (Throwable e)
        {
            String stack = ExceptionUtils.getFullStackTrace(e);
            String error = "Event processing exception; no further events will be published. \r\n" + stack;
            result = new EventResult(error, Collections.<Event>emptyList(), false);
        }
        // See how long it took
        long after = System.currentTimeMillis();
        long time = after - before;
        
        // Get any supplemental data to be recorded
        Serializable data = result.getData();
        // Get the next events to publish
        List<Event> nextEvents = result.getNextEvents();
        // Was it successful?
        boolean wasSuccess = result.isSuccess();
        // Construct the recorded event
        EventRecord recordedEvent = new EventRecord(wasSuccess, before, time, data, event);
        recordedEvent.setChart(chart);
        
        // Check the time taken against the time allowed
        if (time > warnDelay)
        {
            String msg = "Event processing exceeded warning threshold by " + (time - warnDelay) + "ms.";
            recordedEvent.setWarning(msg);
        }
        
        if (logger.isDebugEnabled())
        {
            logger.debug("\n" +
                    "Event processing completed: \n" +
                    "   Owner:     " + serverId + "\n" +
                    "   Test Run:  " + testRunFqn + "\n" +
                    "   Event:     " + event + "\n" +
                    "   Time:      " + time + "\n" +
                    "   Processor: " + processor);
        }

        // Record the event
        try
        {
            resultService.recordResult(recordedEvent);
        }
        catch (Throwable e)
        {
            logger.error("Failed recorded event: " + recordedEvent, e);
        }
        
        // Clean up any locally-store data and remove the event from the queue
        try
        {
            event.cleanData();
            boolean deleted = eventService.deleteEvent(event);
            if (!deleted)
            {
                logger.error("Event was not deleted from the queue: " + event);
            }
        }
        catch (Throwable e)
        {
            logger.error("Failed to remove event from the queue: " + event, e);
        }
        
        // Publish the next events
        for (Event nextEvent : nextEvents)
        {
            // Ensure that any locally-stored data has a data owner attached
            if (nextEvent.getDataKey() != null)
            {
                nextEvent.setDataOwner(serverId);
            }
            try
            {
                eventService.putEvent(nextEvent);
            }
            catch (Throwable e)
            {
                logger.error("Failed to insert event into queue: " + nextEvent);
            }
        }
    }
}
