/*
 * Copyright (C) 2005-2012 Alfresco Software Limited.
 *
 * This file is part of Alfresco
 *
 * Alfresco is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Alfresco is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with Alfresco. If not, see <http://www.gnu.org/licenses/>.
 */
package org.alfresco.bm.event;

import java.util.List;

/**
 * Interface for classes that handle persistence and retrieval of {@link Event events}.
 * <p/>
 * Note that the <i>next</i> event is the event that should be processed
 * next and not necessarily the event at the head of the queue.
 * 
 * @author Derek Hulley
 * @since 1.0
 */
public interface EventService
{
    /**
     * @return                      Returns the number of events in the queue
     */
    long count();
    
    /**
     * Find any event matching the given name, using the paging criteria
     * 
     * @param name                  find events with the given name
     * @param skip                  the number of results to skip
     * @param limit                 the maximum number of results to retrieve 
     * @return                      an event or <tt>null</tt> if not found
     */
    List<Event> findEvents(String name, int skip, int limit);
    
    /**
     * Retrieve the next event in the queue.
     * <p/>
     * This method must update the event queue time to the current time atomically.
     * 
     * @param dataOwner             the identifier of the client queuing events
     * @param latestQueueTime       the event scheduled time must not be greater than this
     * @return                      Returns the next event in the queue or <tt>null</tt>
     */
    Event nextEvent(String dataOwner, long latestScheduledTime, long latestQueueTime);
    
    /**
     * Lock an event for processing by the given owner
     * 
     * @param event                 the event to lock
     * @param lockOwner             the lock owner
     * @return                      Returns the locked event or <tt>null</tt> if it could not
     *                              be locked.
     */
    Event lockEvent(Event event, String lockOwner);

    /**
     * Fetch a specific event by unique ID
     * 
     * @param id                    the event ID
     * @return                      the event or <tt>null</tt> if not found
     */
    Event getEvent(String id);
    
    /**
     * Delete an event from the provider.  This can be done after the event has
     * been fully processed and there is no further requirement for it.
     * <p/>
     * <b>NOTE: The data cannot be accessed after this event.</b>
     * 
     * @param event             the event to delete
     * @return                  <tt>true</tt> if the event was deleted otherwise <tt>false</tt>
     */
    boolean deleteEvent(Event event);
    
    /**
     * Adds an event to the provider. This is done to populate the queue with
     * a new event to be processed.
     * 
     * @param event             the event to add (the {@link Event#getId() event ID} must be <tt>null</tt>
     */
    Event putEvent(Event event);
}
