/*
 * Copyright (C) 2005-2012 Alfresco Software Limited.
 *
 * This file is part of Alfresco
 *
 * Alfresco is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Alfresco is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with Alfresco. If not, see <http://www.gnu.org/licenses/>.
 */
package org.alfresco.bm.event;

import java.io.Serializable;

import org.springframework.data.mongodb.core.MongoTemplate;

import com.mongodb.BasicDBObjectBuilder;
import com.mongodb.DBObject;

/**
 * The record of a processed event.
 * 
 * @author Derek Hulley
 * @since 1.0
 */
public class EventRecord
{
    public static final String FIELD_ID = "id";
    public static final String FIELD_EVENT_NAME = "event.name";
    public static final String FIELD_START_TIME = "startTime";
    public static final String FIELD_SUCCESS = "success";

    public static final String INDEX_EVENT_NAME = "idx_name";
    public static final String INDEX_START_TIME = "idx_start_time";
    public static final String INDEX_SUCCESS = "idx_success";

    private String id;
    private boolean success;
    private long startTime;
    private long startDelay;
    private long time;
    private Event event;
    private Serializable data;
    private String warning;
    private boolean chart = true;

    /**
     * Default constructor for auto-construction by frameworks.
     */
    @SuppressWarnings("unused")
    private EventRecord()
    {
    }
    
    /**
     * Ensure that the MongoDB collection has the required indexes for events.
     * 
     * @param mongo                 connection to MongoDB
     * @param collection            name of DB collection containing events
     */
    public static void checkIndexes(MongoTemplate mongo, String collection)
    {
        DBObject idxName = BasicDBObjectBuilder
                .start(FIELD_EVENT_NAME, 1)
                .add(FIELD_START_TIME, 1)
                .add(FIELD_SUCCESS, 1)
                .get();
        mongo.getDb().getCollection(collection).ensureIndex(idxName, INDEX_EVENT_NAME, false);
        
        DBObject idxStartTime = BasicDBObjectBuilder
                .start(FIELD_START_TIME, 1)
                .add(FIELD_EVENT_NAME, 1)
                .add(FIELD_SUCCESS, 1)
                .get();
        mongo.getDb().getCollection(collection).ensureIndex(idxStartTime, INDEX_START_TIME, false);
        
        DBObject idxSuccess = BasicDBObjectBuilder
                .start(FIELD_SUCCESS, 1)
                .add(FIELD_START_TIME, 1)
                .get();
        mongo.getDb().getCollection(collection).ensureIndex(idxSuccess, INDEX_SUCCESS, false);
    }
    
    /**
     * @param success           <tt>true</tt> if this represents a successful event
     *                          otherwise <tt>false</tt>
     * @param startTime         the time when processing started
     * @param time              the time it took to process the event
     * @param data              any additional data that should be recorded with this event
     * @param event             the event that was processed
     */
    public EventRecord(boolean success, long startTime, long time, Serializable data, Event event)
    {
        this.success = success;
        this.startTime = startTime;
        this.startDelay = (event.getScheduledTime() == 0) ? 0L : (startTime - event.getScheduledTime());
        this.time = time;
        this.data = data;
        this.event = event;
    }

    @Override
    public String toString()
    {
        StringBuilder builder = new StringBuilder();
        builder.append("EventRecord [success=");
        builder.append(success);
        builder.append(", startTime=");
        builder.append(startTime);
        builder.append(", time=");
        builder.append(time);
        builder.append("]");
        return builder.toString();
    }

    public String getId()
    {
        return id;
    }
    public void setId(String id)
    {
        this.id = id;
    }

    public boolean isSuccess()
    {
        return success;
    }
    public void setSuccess(boolean success)
    {
        this.success = success;
    }

    public long getStartTime()
    {
        return startTime;
    }
    public void setStartTime(long startTime)
    {
        this.startTime = startTime;
    }

    public long getStartDelay()
    {
        return startDelay;
    }
    public void setStartDelay(long startDelay)
    {
        this.startDelay = startDelay;
    }

    public long getTime()
    {
        return time;
    }
    public void setTime(long time)
    {
        this.time = time;
    }

    public Event getEvent()
    {
        return event;
    }

    public void setEvent(Event event)
    {
        this.event = event;
    }
    public Serializable getData()
    {
        return data;
    }
    public void setData(Serializable data)
    {
        this.data = data;
    }

    public String getWarning()
    {
        return warning;
    }
    public void setWarning(String warning)
    {
        this.warning = warning;
    }

    public boolean isChart()
    {
        return chart;
    }
    public void setChart(boolean chart)
    {
        this.chart = chart;
    }
}
