/*
 * Copyright (C) 2005-2012 Alfresco Software Limited.
 *
 * This file is part of Alfresco
 *
 * Alfresco is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * Alfresco is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with Alfresco. If not, see <http://www.gnu.org/licenses/>.
 */
package org.alfresco.bm.event;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.BeanNameAware;

/**
 * Basic services for the {@link EventProcessor event processor} implementations
 * <p/>
 * <b>Read {@link EventProcessor 'good practice'} notes for documenting behaviour.
 * 
 * @author Derek Hulley
 * @since 1.0
 */
public abstract class AbstractEventProcessor implements EventProcessor, BeanNameAware
{
    protected static final long DEFAULT_WARN_DELAY = Long.MAX_VALUE;
    protected static final boolean DEFAULT_CHART = Boolean.TRUE;

    /** Resource for derived classes to use for logging */
    protected Log logger = LogFactory.getLog(this.getClass());
    private EventProcessorRegistry registry;
    private List<String> eventNames;
    private long warnDelay;
    private boolean chart;
    
    /**
     * Default constructor
     */
    public AbstractEventProcessor()
    {
        eventNames = new ArrayList<String>(1);
        warnDelay = DEFAULT_WARN_DELAY;
        chart = DEFAULT_CHART;
    }

    @Override
    public void setBeanName(String beanName)
    {
        String eventName = beanName.replaceFirst(Event.EVENT_BEAN_PREFIX, "");
        setEventName(eventName);
    }

    /**
     * Set the single event name that this processor can handle
     */
    public void setEventName(String eventName)
    {
        setEventNames(Collections.singletonList(eventName));
    }
    
    /**
     * Set the names of the events that this processor can handle 
     */
    public void setEventNames(List<String> eventNames)
    {
        this.eventNames = eventNames;
    }

    /**
     * Inject the registry that this instance will {@link #register() register} with.
     */
    public synchronized void setRegistry(EventProcessorRegistry registry)
    {
        this.registry = registry;
    }

    /**
     * @return              the processing time after which an warning is logged (default {@link #DEFAULT_WARN_DELAY})
     */
    @Override
    public long getWarnDelay()
    {
        return warnDelay;
    }

    /**
     * @param warnDelay     the time allowed for processing before a warning is logged
     * @see #getWarnDelay()
     */
    public void setWarnDelay(long warnDelay)
    {
        this.warnDelay = warnDelay;
    }

    /**
     * @return              <tt>true</tt> if results from this processor must be included
     *                      in generated charts
     */
    public boolean isChart()
    {
        return chart;
    }

    /**
     * @param chart         <tt>true</tt> to tag results for charting
     * @see #setChart(boolean)
     */
    public void setChart(boolean chart)
    {
        this.chart = chart;
    }

    /**
     * Register this instance for the event names that are handled.
     */
    public synchronized void register()
    {
        if (registry == null)
        {
            // Nothing to do
            return;
        }
        for (String eventName : eventNames)
        {
            registry.register(eventName, this);
        }
    }
}
